import Data.List(zip4, maximumBy)
import System.Random(randomRIO)
import System.Console.ANSI(clearScreen)
import Data.Function(on)

transpose :: [[Int]] -> [[Int]]
transpose [r1, r2, r3, r4] = map (\(x1,x2,x3,x4) -> [x1,x2,x3,x4]) $ zip4 r1 r2 r3 r4
transpose _ = error "can not transpose a non 4x4 matrix"

moveDown :: [[Int]] -> [[Int]]
moveDown board = reverse $ moveUp $ reverse board

moveUp :: [[Int]] -> [[Int]]
moveUp board = transpose $ moveLeft $ transpose board 

moveRight :: [[Int]] -> [[Int]]
moveRight board = map reverse $ moveLeft $ map reverse board

moveLeft :: [[Int]] -> [[Int]]
moveLeft board = map moveRow board where 
  moveRow :: [Int] -> [Int]
  moveRow row = let merged = merge [x | x <- row, x /= 0] [] in 
    reverse $ replicate (4-length merged) 0 ++ merged
  merge :: [Int] -> [Int] -> [Int]
  merge [] acc = acc
  merge [x] acc = x:acc
  merge (f:s:xs) acc
    | f == s = merge xs (f*2:acc)
    | otherwise = merge (s:xs) (f:acc)

canMove :: [[Int]] -> Bool
canMove board = any checkRow board where
  checkRow :: [Int] -> Bool
  checkRow row = any (\(a, b) -> a == b || a == 0 || b == 0) $ zip row (tail row)


fill :: [[Int]] -> Int -> Int -> Int -> [[Int]]
fill board x y v = prev ++ (newRow : next) where
  (prev, row:next) = splitAt x board
  newRow = take y row ++ v : drop (y+1) row 

spawn :: [[Int]] -> IO [[Int]]
spawn board = do 
  let slots = [ (x, y) | (x, row) <- zip [0..] board, (y, val) <- zip [0..] row, val == 0]
  case length slots of
    0 -> pure board
    _ -> do
      val <- randomRIO (1, 10::Int) >>= pure . (\x -> if x == 1 then 4 else 2)
      (xpos, ypos) <- randomRIO (0, length slots-1) >>= pure . (slots !!)
      return $ fill board xpos ypos val

printBoard :: [[Int]] -> IO ()
printBoard board = clearScreen >> mapM_ printRow board >> putStrLn "" where
    printRow row = putStrLn $ tail $ foldr printNum "" $ map show row
    printNum num out = (replicate (5 - (length num)) ' ')++num++out


heuristic :: [[Int]] -> Double
heuristic board
  | canMove board = snakeSumHeu - snakeHeadHeu
  | otherwise = read "-Infinity" where
    snakeHeadHeu = if head snake == snakeMax then 0 else (abs $ head snake - snakeMax) ** 2
    snakeSumHeu = foldl (\acc (i, x) -> acc + x/10**i) 0 $ zip [0..] snake
    snakeMax = maximum snake
    snake = map fromIntegral $ concat $ map (\(i, row) -> if i `mod` (2::Int) == 0 then reverse row else row) $ zip [0..] $ transpose board 

search :: [[Int]] -> Int -> Bool -> Double
search board depth onMove
  | depth == 0 || (onMove && not (canMove board)) = heuristic board
  | onMove = maximum $ heuristic board:map (\action -> search (action board) (depth-1) False) actions
  | otherwise = sum $ map fillOne choices
  where
    fillOne (x,y,(v, p)) = p * (search (fill board x y v) (depth-1) True) / fromIntegral (length slots)
    choices = [(x, y, vp) | (x, y) <- slots, vp <- [(2, 0.9), (4, 0.1)]::[(Int, Double)] ]
    slots = [ (x, y) | (x, row) <- zip [0..] board, (y, val) <- zip [0..] row, val == 0]
    actions = [moveUp, moveLeft, moveRight, moveDown]

canPlay :: [[Int]] -> Bool
canPlay board = (canMove board) || (canMove $ transpose board)

play :: [[Int]] -> IO ()
play board
  | elem 2048 (concat board) = printBoard board >> putStrLn "Success."
  | canPlay board = do 
    printBoard board 
    case elem 0 (concat nextBoard) of
      False -> putStrLn "Lost."
      _ ->
        spawn nextBoard >>= play
  | otherwise = printBoard board >> putStrLn "Lost."
  where
    nextBoard = snd $ maximumBy (compare `on` fst) $ map helper actions where
      helper = \action -> let next = action board in (search next 4 False, next)
    actions = [moveUp, moveLeft, moveRight, moveDown]
    
main :: IO ()
main = do
  pure [[0,0,0,0], [0,0,0,0], [0,0,0,0], [0,0,0,0]] >>= spawn >>= spawn >>= play


