{-# LANGUAGE BangPatterns #-}
{--
ref: 
https://stackoverflow.com/questions/12547160/how-does-the-dpll-algorithm-work
https://en.wikipedia.org/wiki/DPLL_algorithm
https://gist.github.com/adrianlshaw/1807739
https://www.cs.cmu.edu/~15414/f17/lectures/10-dpll.pdf
--}

module BoolSat where

import Data.Maybe
import Control.Parallel.Strategies

type Literal = Integer
type Clause = [Literal]
type Formula = [Clause]
type Record = [Literal]

{-
  SolverState holds the formula and the record. The formula is the CNF-SAT 
  problem which is modified as different literals are assigned. The record
  holds the assignments that are 'true'
-}
data SolverState = SolverState {formula :: !Formula
                               , record :: !Record
                               } deriving (Show)


{-Sequential DPLL algorithm that uses unitpropagation and backtracking-}
seqDpll :: SolverState -> Maybe Record
seqDpll s 
	| null cnf = return rec
	| otherwise = do 
		l <- chooseLiteral cnf
		case seqDpll (SolverState (eliminateLiteral cnf l) (l:rec)) of 
			Just res -> return res
			Nothing -> seqDpll $ SolverState (eliminateLiteral cnf (-l)) ((-l):rec)
	where 
		state' = unitpropagate s 
		cnf = formula state'
		rec = record state'

{-
	[DEPRECATED]: first try: parallelize dpll.
-}
trial_dpllPar :: (Ord a, Num a) => a -> SolverState -> Maybe Record 
trial_dpllPar _ (SolverState [] rec) = Just rec
trial_dpllPar i s
	| null cnf = return rec
	| otherwise = do
		case getUnit cnf of
			Just u -> trial_dpllPar i $ SolverState (eliminateLiteral cnf u) (u:rec)
			Nothing ->
				let 
				  dlit = unwrapMaybe $ chooseLiteral cnf
				  trueBranch = trial_dpllPar (i-1) 
				               (SolverState (eliminateLiteral cnf dlit) (dlit:rec))
				  falseBranch = trial_dpllPar (i-1) 
				               (SolverState (eliminateLiteral cnf (-dlit)) ((-dlit):rec))
				in if i > 0 then
					runEval $ do 
						x <- rpar falseBranch 
						return (case trueBranch of 
							Nothing -> x
							Just r -> return r)
				   else
				   	case trueBranch of 
				   		Nothing -> falseBranch 
				   		Just r -> return r

	where
		state' = unitpropagate s 
		cnf = formula state'
		rec = record state' 

{-
  Parallel DPLL that does not have any depth. It parallelizes the dpll on the branching part
  i.e. the last line of the dpll pseudocode. Computes the falseBranch in par with the true one.
-}
parDpll :: SolverState -> Maybe Record
parDpll s 
	| null cnf = return rec 
  	| otherwise = do 
  		l <- chooseLiteral cnf
  		let trueBranch = parDpll (SolverState (eliminateLiteral cnf l) (l:rec))
  		let	falseBranch = parDpll (SolverState (eliminateLiteral cnf (-l)) ((-l):rec))
  		runEval $ do 
  			x <- rpar falseBranch 
  			return (case trueBranch of
  				Nothing -> x
  				Just r -> return r)

  	where 
  		state' = unitpropagate s 
		cnf = formula state'
		rec = record state' 


---------------------------------
{-
  Parallel DPLL that has a depth param. It parallelizes the dpll on the branching part
  i.e. the last line of the dpll pseudocode. Computes the falseBranch in par with the true one.
-}
parDpll3 :: (Ord t, Num t) => t -> SolverState -> Maybe Record
parDpll3 i s 
	| null cnf = return rec
  	| otherwise = do 
  		l <- chooseLiteral cnf
  		let trueBranch = parDpll3 (i-1) (SolverState (eliminateLiteral cnf l) (l:rec))
  		let	falseBranch = parDpll3 (i-1) (SolverState (eliminateLiteral cnf (-l)) ((-l):rec))
  		if i > 0 then
	  		runEval $ do 
	  			x <- rpar falseBranch 
	  			return (case trueBranch of
	  				Nothing -> x
	  				Just r -> return r)
	  	else
	  		case trueBranch of
	  			Nothing -> falseBranch
	  			Just r -> return r

  	where 
  		state' = unitpropagate s 
		cnf = formula state'
		rec = record state'
-------------------------------------


{-More info on unit propagation in the project report-}
unitpropagate :: SolverState -> SolverState
unitpropagate (SolverState cnf rec) = 
	case getUnit cnf of
		Nothing -> SolverState cnf rec
		Just u -> unitpropagate $ SolverState (eliminateLiteral cnf u) (u:rec) 

{-Checks for a literal in the formula and returns it (or Nothing otherwise)-}
chooseLiteral :: Formula -> Maybe Literal
chooseLiteral cnf = listToMaybe . concat $ cnf


{-
  Checks for a clause with only one literal and returns the literal, other-
  wise it returns Nothing
-}
getUnit :: Formula -> Maybe Literal
getUnit xs = listToMaybe [x | [x] <- xs]


{-
  This does what was described under the section of unit propagation in the 
  report. That is, given a literal a to be simplified with, it: " (1) removes 
  instances of other clauses that contain the literal and (2) removes 
  instances of the literal's complement in other clauses. 
  Proof: (a OR _) = a, (a OR (NOT a)) = a
 -}
eliminateLiteral :: Formula -> Literal -> Formula
eliminateLiteral cnf l = [simplClause x l | x <- cnf, not (elem l x)]
	where
		simplClause c lit = filter (/= -lit) c


{-solver for sequential; takes cnf formula as input, together with empty list
that will hold the output-}
seqDpllSolve :: [[Integer]] -> Maybe [Integer]
seqDpllSolve = seqDpll . flip SolverState []

{-solvers for parallel; takes cnf formula as input, together with empty list
that will hold the output-}
parDpllTrialSolve :: (Ord a, Num a) => a -> [[Integer]] -> Maybe [Integer]
parDpllTrialSolve i = trial_dpllPar i . flip SolverState []

parDpllSolve :: [[Integer]] -> Maybe [Integer]
parDpllSolve = parDpll . flip SolverState []


parDpllSolve3 :: (Ord a, Num a) => a -> [[Integer]] -> Maybe [Integer]
parDpllSolve3 i = parDpll3 i . flip SolverState []

{-obtaining the value in the Maybe type-}
unwrapMaybe :: Maybe a -> a 
unwrapMaybe (Just n) = n
unwrapMaybe Nothing = error $ "Nothing is returned here"










