/* Ocamlyacc parser for nodable */

%{ open Ast

let scope = ref 1

let parse_error err =
	print_endline err;
	flush stdout

%}

%token SEMI COMMA
%token LPAREN RPAREN LBRACE RBRACE LSQUARE RSQUARE ASSIGN DOT DOLLAR

%token PLUS MINUS TIMES DIVIDE MODULO
%token NOT OR AND LT GT LEQ GEQ EQ NEQ 
%token NEW

%token INT FLOAT BOOL STRING NODE LIST VOID
%token <int> INT_LIT
%token  <string> FLT_LIT STR_LIT ID
%token <bool> BOOL_LIT

%token RETURN FOR IF ELIF ELSE WHILE
%token EOF

%start program
%type <Ast.program> program

%nonassoc NOELSE
%nonassoc ELSE
%nonassoc LPAREN LSQUARE LBRACE
%nonassoc RPAREN RSQUARE RBRACE

%left COMMA
%right ASSIGN
%left OR
%left AND
%left DOT
%left EQ NEQ
%left LEQ GEQ LT GT 
%left PLUS MINUS
%left TIMES DIVIDE MODULO
%right NOT NEG

%%

program: 
  decls EOF { $1 }

decls:
	/* nothing */ 			{ ([], [])               }
	| decls vdecl 			{ (($2 :: fst $1), snd $1) }
	| decls fdecl 			{ (fst $1, ($2 :: snd $1)) }
	| decls glb_vdecl 	{ (($2 :: fst $1), snd $1) }


fdecl:
   typ ID LPAREN formals_opt RPAREN LBRACE stmt_list RBRACE
     { 
			{ typ = $1;
				fname = $2;
				formals = $4;
				body = List.rev $7 
			} 
		}

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    typ ID                   { [($1,$2)]     }
  | formal_list COMMA typ ID { ($3,$4) :: $1 }

expr_opt: 
	/* nothing */			{ Noexpr }
	| expr					{ $1 }

literal:
	ID										{ Id($1) }
	| INT_LIT								{ Lit_Int($1) }
	| FLT_LIT								{ Lit_Flt($1) }
	| STR_LIT								{ Lit_Str($1) }
	| BOOL_LIT								{ Lit_Bool($1) }
	| LPAREN expr RPAREN            		{ $2 }
	| ID LPAREN args_opt RPAREN  			{ Call($1, $3) }
	| literal LSQUARE expr RSQUARE    		{ List_Access($1, $3) }
	| literal DOT ID 						{ Attr($1, $3) }

expr:
	literal 										{ $1 }
	| node_expr 								{ $1 } 	
	| expr PLUS expr						{ Binop ($1, Add, $3) }
	| expr MINUS expr						{ Binop ($1, Sub, $3) }
	| expr DIVIDE expr					{ Binop ($1, Div, $3) }
	| expr TIMES expr 					{ Binop ($1, Mult, $3) }
	| expr MODULO expr	 				{ Binop ($1, Mod, $3) }
	| expr LT expr							{ Binop ($1, Less, $3) }
	| expr GT expr							{ Binop ($1, Greater, $3) }
	| expr LEQ expr							{ Binop ($1, Leq, $3) }
	| expr GEQ expr							{ Binop ($1, Geq, $3) }
	| expr EQ expr							{ Binop ($1, Eq, $3) }
	| expr NEQ expr							{ Binop ($1, Neq, $3) }
	| expr AND expr							{ Binop ($1, And, $3) }
	| expr OR expr							{ Binop ($1, Or, $3) }
	| MINUS expr %prec NOT 			{ Unop(Neg, $2)      }
	| NOT expr         					{ Unop(Not, $2)      }
	| ID ASSIGN expr   					{ Assign($1, $3)     }
	| LSQUARE args_opt RSQUARE	{ Lit_List($2) }

node_expr:
	DOLLAR literal			{ Lit_Node($2) }

args_opt:
  /* nothing */ { [] }
  | args_list  { List.rev $1 }

args_list:
    expr                  { [$1] }
  | args_list COMMA expr 	{ $3 :: $1 }

typ:
	  INT							{ Int }
	| FLOAT						{ Float }
	| BOOL						{ Bool }
	| STRING					{ String }
	| VOID 						{ Void }
	| NODE LT typ GT	{ Node($3) }
	| LIST LT typ GT 	{ List($3) }

vdecl:
	typ ID SEMI			{ ($1, $2) }

glb_vdecl: 
	vdecl SEMI{$1}

stmt_list:
	/* nothing */			{ []} 
	| stmt_list stmt 		{ $2 :: $1 } 

stmt:
	expr SEMI 																									{ Expr($1) } 
	| typ ID SEMI 																							{ Declare($1, $2, Noexpr) }
	| typ ID ASSIGN expr SEMI																				{ Declare($1, $2, Assign($2, $4)) }
	| RETURN expr_opt SEMI 																			{ Return ($2) } 
	| LBRACE stmt_list RBRACE 																	{ Block(List.rev $2) }
	| IF LPAREN expr RPAREN stmt %prec NOELSE 									{ If($3, $5, Block([])) }
	| IF LPAREN expr RPAREN stmt ELSE stmt 											{ If($3, $5, $7) }
	| WHILE LPAREN expr RPAREN stmt															{ While($3, $5) }
	| FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt 	{ For($3, $5, $7, $9) }	