(* Semantically-checked Abstract Syntax Tree and functions for printing it *)

open Ast

type sunt = 
  SUnit of string


type sexpr = typ * sx
and sx =
  SIntLit of int
  | SFloatLit of string
  | SBoolLit of bool
  | SStringLit of string
  | SId of string
  | SBinop of sexpr * op * sexpr
  | SUnop of uop * sexpr
  | SAssign of sexpr * sexpr
  | SFunctionCall of string * sexpr list
  | SArray of sexpr list 
  | SArrayAccess of sexpr * sexpr 
  | SNoexpr

type sstmt =
    SBlock of sstmt list
  | SExpr of sexpr
  | SReturn of sexpr
  | SIf of sexpr * sstmt * sstmt
  | SFor of sexpr * sexpr * sexpr * sstmt
  | SWhile of sexpr * sstmt
  | SDAssign of typ * string * string * sexpr

type sfunc_decl = {
    sreturn_type : typ;
    sreturn_unit : uni;
    sfunc_identifier : string;
    sfunc_formals : ubind list;
    sfunc_stmts: sstmt list;
  }


type sprogram = unit_decl list * ubind list * sfunc_decl list

(* Pretty-printing functions *)

let rec string_of_sexpr (t, e) =
  "(" ^ string_of_typ t ^ " : " ^ (match e with
    SIntLit(l) -> string_of_int l
  | SBoolLit(true) -> "true"
  | SBoolLit(false) -> "false"
  | SFloatLit(l) -> l
  | SStringLit(l) -> l
  | SId(s) -> s
  | SBinop(e1, o, e2) ->
      string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
  | SUnop(o, e) -> string_of_uop o ^ string_of_sexpr e
  | SAssign(v, e) -> string_of_sexpr v ^ " = " ^ string_of_sexpr e
  | SFunctionCall(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_sexpr el) ^ ")"
  | SArray(el) -> String.concat ", " (List.map string_of_sexpr el)
  | SArrayAccess(e1, e2) -> string_of_sexpr e1 ^ "[" ^ string_of_sexpr e2 ^ "]"
        
  | SNoexpr -> ""
				  ) ^ ")"				     

let rec string_of_sstmt = function
    SBlock(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_sstmt stmts) ^ "}\n"
  | SExpr(expr) -> string_of_sexpr expr ^ ";\n";
  | SReturn(expr) -> "return " ^ string_of_sexpr expr ^ ";\n";
  | SIf(e, s, SBlock([])) ->
      "if (" ^ string_of_sexpr e ^ ")\n" ^ string_of_sstmt s
  | SIf(e, s1, s2) ->  "if (" ^ string_of_sexpr e ^ ")\n" ^
      string_of_sstmt s1 ^ "else\n" ^ string_of_sstmt s2
  | SFor(e1, e2, e3, s) ->
      "for (" ^ string_of_sexpr e1  ^ " ; " ^ string_of_sexpr e2 ^ " ; " ^
      string_of_sexpr e3  ^ ") " ^ string_of_sstmt s
  | SWhile(e, s) -> "while (" ^ string_of_sexpr e ^ ") " ^ string_of_sstmt s
  | SDAssign(t, u, v, e) -> string_of_typ t ^ " " ^ u ^ " " ^ v ^ " = " ^ string_of_sexpr e ^ ";\n" 


let get_3_3 t = match t with 
(_, _, e) -> e;;

let string_of_sfdecl fdecl =
  string_of_typ fdecl.sreturn_type ^ " " ^ "func " ^ 
  fdecl.sfunc_identifier ^ "(" ^ String.concat ", " (List.map get_3_3 fdecl.sfunc_formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_sstmt fdecl.sfunc_stmts) ^
  "}\n"

let string_of_sprogram (udecls, vars, funcs) =
  String.concat "" (List.map string_of_udecl udecls) ^ "\n" ^
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_sfdecl funcs)
