(* open Printf *)

type operator = Add | Sub | Mul | Div | Seq | Lt | Gt | Mod | Lte | Gte
			| Eq | Ne | And | Or

type unary_operator = Not

type typ = Int | Float | Boolean | Void | String | Array

type expr =
    Liti of int
  | Litf of string
  | Litb of bool
  | Lits of string
  | LitArray of typ * expr
  | Assign of string * expr
  | ArrayIndexAssign of string * expr * expr
  | ArrayIndexAccess of string * expr
  | Id of string
  | Binop of expr * operator * expr
  | Uniop of unary_operator * expr
  | Call of string * expr list
  | Noexpr


type stmt =
    Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt

type bind = typ * string * expr

type func_decl = {
    typ : typ;
    fname : string;
    formals : bind list;
    locals : bind list;
    body : stmt list;
}

type program = bind list * func_decl list * stmt list


(* Pretty-printing functions *)

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mul -> "*"
  | Div -> "/"
  | Eq -> "=="
  | Ne -> "!="
  | Lt -> "<"
  | Lte -> "<="
  | Gt -> ">"
  | Gte -> ">="
  | And -> "&&"
  | Or -> "||"
  | Mod -> "%"

let string_of_typ = function
    Int -> "int"
  | Boolean -> "boolean"
  | Float -> "float"
  | Void -> "void"
  | String -> "string"
  | Array -> "array"

let string_of_uop = function
    Not -> "not"

let rec string_of_expr = function
    Liti(l) -> string_of_int l
  | Litf(l) -> l (* sprintf "%f" l *)
  | Litb(true) -> "true"
  | Litb(false) -> "false"
  | Lits(l) -> l
  | LitArray(t, size) -> "array of type " ^ string_of_typ t ^ " and size " ^ string_of_expr size
  | Id(s) -> s
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Uniop(o, e) -> string_of_uop o ^ string_of_expr e
  | Assign(v, e) -> v ^ " = " ^ string_of_expr e
  | ArrayIndexAssign(s, i, e) -> "array assignment of " ^ string_of_expr e ^ " at index " ^ string_of_expr i ^ " for array " ^ s
  | ArrayIndexAccess(s, i) -> "array " ^ s ^ "at index " ^ string_of_expr i
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr -> ""

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let string_of_vdecl (t, id, e) = string_of_typ t ^ " " ^ id ^ ";\n"

let string_of_fdecl fdecl =
  string_of_typ fdecl.typ ^ " " ^
  fdecl.fname ^ "(" ^ String.concat ", " (List.map string_of_vdecl fdecl.formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.locals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_program (vars, funcs, statements) =
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs) ^ "\n" ^
  String.concat "\n" (List.map string_of_stmt statements)
