(* Ocamllex scanner for Pixel *)
{ open Parser }

let digit = ['0' - '9']
let digits = digit+

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf } (* Whitespace *)
| "/*"     { comment lexbuf }           (* Comments *)
| "//" 	   { singlecomment lexbuf}
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| ';'      { SEMI }
| '['      { LBRACK }
| ']'      { RBRACK }
| ','      { COMMA }
| '.'      { DOT }
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '='      { ASSIGN }
| "**"     { EXP }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "&&"     { AND }
| "||"     { OR }
| "!"      { NOT }
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "fun"    { FUN }
| "::"     { DBLCOLON }
| "matrix" { MATRIX }
| "image"  { IMAGE }
| "while"  { WHILE }
| "return" { RETURN }
| "int"    { INT }
| "float"   { FLOAT }
| "void"   { VOID }
| "grayscale" { GRAYSCALE }
| "red"    { RED }
| "green"  { GREEN }
| "blue"   { BLUE }
| "rows"   { ROWS }
| "cols"   { COLS }
| digits as lxm { LITERAL(int_of_string lxm) }
| digits '.'  digit* ( ['e' 'E'] ['+' '-']? digits )? as lxm { FLIT(lxm) }
| ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']*     as lxm { ID(lxm) }
| '"' ([^ '"']* as lxm) '"' { STR_LITERAL(lxm) }
| eof { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and singlecomment = parse
  "\n" { token lexbuf }
| _    { singlecomment lexbuf }

and comment = parse
  "*/" { token lexbuf }
| _    { comment lexbuf }