open Ast

type sexpr = typ * sx
and sx =
    SLiteral of int
  | SFliteral of string
  | SMLiteral of sexpr list * int * int
  | SStrLiteral of string
  | SId of string
  | SBinop of sexpr * op * sexpr
  | SUnop of uop * sexpr
  | SAssign of string * sexpr
  | SCall of string * sexpr list
  | SNoexpr
  | SMatrixAccess of string * sexpr * sexpr
  | SImageRedAccess of string
  | SImageGreenAccess of string
  | SImageBlueAccess of string
  | SImageGrayscaleAccess of string
  | SMatrixRows of string
  | SMatrixCols of string

type sstmt =
    SBlock of sstmt list
  | SExpr of sexpr
  | SReturn of sexpr
  | SIf of sexpr * sstmt * sstmt
  | SFor of sexpr * sexpr * sexpr * sstmt
  | SWhile of sexpr * sstmt
  | SVariable of typ * string * sexpr
  | SMatrixAssign of typ * string * sexpr * int * int
  | SMatrixAccessAssign of string * sexpr * sexpr * sexpr

type sbind = typ * string * sexpr

type sfunc_decl = {
  styp: typ;
  sfname: string;
  sformals: sbind list;
  sbody: sstmt list;
}

type sdecls =
  | SStmtDecl of sstmt
  | SFuncDecl of sfunc_decl

type sprogram = sdecls list

(* Pretty printing *)
let rec string_of_sexpr (t, e) =
  "(" ^ string_of_typ t ^ " : " ^ (match e with
    SLiteral(l) -> string_of_int l
  | SFliteral(l) -> l
  | SStrLiteral(s) -> "\"" ^ s ^ "\""
  | SId(s) -> s
  | SMLiteral(l, _, _) -> "[" ^ String.concat ", " (List.map string_of_sexpr l) ^ "]"
  | SBinop(e1, o, e2) ->
      string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
  | SUnop(o, e) -> string_of_uop o ^ string_of_sexpr e
  | SAssign(v, e) -> v ^ " = " ^ string_of_sexpr e
  | SCall(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_sexpr el) ^ ")"
  | SNoexpr -> ""
  | SMatrixAccess(v, e1, e2) -> v ^ "[" ^ string_of_sexpr e1 ^ ", " ^ string_of_sexpr e2 ^ "]"
  | SImageRedAccess(v) -> v ^ ".red"
  | SImageGreenAccess(v) -> v ^ ".green"
  | SImageBlueAccess(v) -> v ^ ".blue"
  | SImageGrayscaleAccess(v) -> v ^ ".grayscale"
  | SMatrixRows(v) -> v ^ ".rows"
  | SMatrixCols(v) -> v ^ ".cols"
  ) ^ ")"

let rec string_of_sstmt = function
    SBlock(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_sstmt stmts) ^ "}\n"
  | SExpr(expr) -> string_of_sexpr expr ^ ";\n";
  | SReturn(expr) -> "return " ^ string_of_sexpr expr ^ ";\n";
  | SIf(e, s, SBlock([])) -> "if (" ^ string_of_sexpr e ^ ")\n" ^ string_of_sstmt s
  | SIf(e, s1, s2) ->  "if (" ^ string_of_sexpr e ^ ")\n" ^
      string_of_sstmt s1 ^ "else\n" ^ string_of_sstmt s2
  | SFor(e1, e2, e3, s) ->
      "for (" ^ string_of_sexpr e1  ^ "; " ^ string_of_sexpr e2 ^ "; " ^
      string_of_sexpr e3  ^ ") " ^ string_of_sstmt s
  | SWhile(e, s) -> "while (" ^ string_of_sexpr e ^ ") " ^ string_of_sstmt s
  | SVariable(t, v, e) -> if string_of_sexpr e = ""
      then string_of_typ t ^ " " ^ v ^ ";\n" (* No value assigned *)
      else string_of_typ t ^ " " ^ string_of_sexpr e ^ ";\n" (* w/ value assigned *)
  | SMatrixAssign(t, v, e, _, _) -> "matrix::" ^ string_of_typ t ^ " " ^ v ^ " = " ^ string_of_sexpr e ^ ";\n"
  | SMatrixAccessAssign(v, e1, e2, e3) -> v ^ "[" ^ string_of_sexpr e1 ^ ", " ^ string_of_sexpr e2 ^ "] = " ^ string_of_sexpr e3 ^ ";"


let string_of_svdecl (t, id, e) = if string_of_sexpr e = ""
then string_of_typ t ^ " " ^ id ^ ";\n"
else string_of_typ t ^ " " ^ string_of_sexpr e ^ ";\n"

let string_of_sfdecl fdecl =
  "fun::" ^ string_of_typ fdecl.styp ^ " " ^ fdecl.sfname ^ "(" ^
    String.concat ", " (List.map string_of_svdecl fdecl.sformals) ^
  ")\n{\n" ^ String.concat "" (List.map string_of_sstmt fdecl.sbody) ^
  "}\n"

let string_of_sdecl = function
  | SStmtDecl(s) -> string_of_sstmt s
  | SFuncDecl(f) -> string_of_sfdecl f

let string_of_sprogram sdecls =
  String.concat "" (List.map string_of_sdecl sdecls) ^ "\n"