/* Ocamyacc parser for Pixel */
%{
  open Ast
%}

%token SEMI LPAREN RPAREN LBRACE RBRACE LBRACK RBRACK COMMA DOT
%token PLUS MINUS TIMES DIVIDE ASSIGN EXP
%token EQ LT LEQ GT GEQ IF FOR WHILE FUN DBLCOLON
%token NOT NEQ AND OR ELSE RETURN
%token INT FLOAT MATRIX IMAGE VOID STRING
%token GRAYSCALE RED GREEN BLUE ROWS COLS
%token <int> LITERAL
%token <string> ID FLIT STR_LITERAL
%token EOF

%start program
%type <Ast.program> program

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left EXP
%left PLUS MINUS
%left TIMES DIVIDE
%right NOT

%%

program:
  decls EOF { List.rev $1 }

decls:
    /* nothing */ { [] }
  | decls fdecl   { FuncDecl($2)::$1 }

fdecl:
  FUN DBLCOLON typ ID LPAREN formals_opt RPAREN LBRACE stmt_list RBRACE
  { {
    typ = $3;
    fname = $4;
    formals = $6;
    body = List.rev $9
  } }

formals_opt:
    /* nothing */ { [] }
  | formal_list   { $1 }

formal_list:
    typ ID                   { [($1, $2)] }
  | formal_list COMMA typ ID { ($3, $4) :: $1 }

typ:
    INT    { Int }
  | FLOAT  { Float }
  | MATRIX { Matrix }
  | IMAGE  { Image }
  | VOID   { Void }
  | STRING { String }

stmt_list:
    /* nothing */  { [] }
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI           { Expr $1 }
  | RETURN expr_opt SEMI { Return $2 }
  | LBRACE stmt_list RBRACE { Block(List.rev $2) }
  | IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt { If($3, $5, $7) }
  | FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt
                                         { For($3, $5, $7, $9) }
  | WHILE LPAREN expr RPAREN stmt { While($3, $5) }
  | typ ID SEMI { Variable($1, $2, Noexpr) }
  | typ ID ASSIGN expr SEMI { Variable($1, $2, Assign($2, $4)) }
  | MATRIX DBLCOLON typ ID ASSIGN expr SEMI
                                         { MatrixAssign($3, $4, $6) }
  | ID LBRACK expr COMMA expr RBRACK ASSIGN expr SEMI
                               { MatrixAccessAssign($1, $3, $5, $8) }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    LITERAL { Literal($1) }
  | FLIT { Fliteral($1) }
  | ID { Id($1) }
  | STR_LITERAL { StrLiteral($1) }
  | LBRACK matrix_body RBRACK { MLiteral($2) }
  | expr PLUS expr { Binop($1, Add, $3) }
  | expr MINUS expr { Binop($1, Sub, $3) }
  | expr TIMES expr { Binop($1, Mult, $3) }
  | expr DIVIDE expr { Binop($1, Div, $3) }
  | expr EXP expr { Binop($1, Exp, $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq, $3) }
  | expr LT     expr { Binop($1, Less, $3) }
  | expr LEQ    expr { Binop($1, Leq, $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GEQ    expr { Binop($1, Geq, $3) }
  | expr AND    expr { Binop($1, And, $3) }
  | expr OR     expr { Binop($1, Or, $3) }
  | MINUS expr %prec NOT { Unop(Neg, $2) }
  | NOT expr { Unop(Not, $2) }
  | ID LPAREN args_opt RPAREN { Call($1, $3) }
  | LPAREN expr RPAREN { $2 }
  | ID ASSIGN expr   { Assign($1, $3) }
  | ID DOT RED  { ImageRedAccess($1) }
  | ID DOT GREEN  { ImageGreenAccess($1) }
  | ID DOT BLUE  { ImageBlueAccess($1) }
  | ID DOT ROWS { MatrixRows($1) }
  | ID DOT COLS { MatrixCols($1) }
  | ID DOT GRAYSCALE { ImageGrayscaleAccess($1) }
  | ID LBRACK expr COMMA expr RBRACK { MatrixAccess($1, $3, $5) }

args_opt:
    /* nothing */ { [] }
  | args_list     { List.rev $1 }

args_list:
    expr                 { [$1] }
  | args_list COMMA expr { $3 :: $1 }

matrix_body:
    LBRACK args_list RBRACK { [MLiteral(List.rev $2)] }
  | LBRACK args_list RBRACK COMMA matrix_body { MLiteral(List.rev $2) :: $5 }