type op = Add | Sub | Mult | Div | Equal | Neq | Less | Leq | Greater | Geq
        | And | Or | Exp

type uop = Neg | Not

type typ = Int | Float | Void | String | Image | Matrix

type expr =
    Literal of int
  | Fliteral of string
  | MLiteral of expr list * int * int
  | StrLiteral of string
  | Id of string
  | Binop of expr * op * expr
  | Unop of uop * expr
  | Assign of string * expr
  | Call of string * expr list
  | Noexpr
  | MatrixAccess of string * expr * expr 
  | ImageRedAccess of string
  | ImageGreenAccess of string
  | ImageBlueAccess of string
  | ImageGrayscaleAccess of string
  | MatrixRows of string
  | MatrixCols of string

type stmt =
    Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  | Variable of typ * string * expr 
  | MatrixAssign of typ * string * expr 
  | MatrixAccessAssign of string * expr * expr * expr (*Incomplete*)

type bind = typ * string * expr

type func_decl = {
  typ: typ;
  fname: string;
  formals: bind list;
  body: stmt list;
}

type decls =
  | StmtDecl of stmt
  | FuncDecl of func_decl

type program = decls list (*Go over how this differs from  `bind list * func_decl list`*)

(* Pretty printing *)
let string_of_typ = function
    Int -> "int"
  | Float -> "float"
  | Void -> "void"
  | String -> "string"
  | Image -> "image"
  | Matrix -> "matrix"

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"
  | Exp -> "**"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"

let rec string_of_expr = function
    Literal(l) -> string_of_int l
  | Fliteral(l) -> l
  | StrLiteral(s) -> "\"" ^ s ^ "\""
  | Id(s) -> s
  | MLiteral(l, _, _) -> "[" ^ String.concat ", " (List.map string_of_expr l) ^ "]"
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Unop(o, e) -> string_of_uop o ^ string_of_expr e
  | Assign(v, e) -> v ^ " = " ^ string_of_expr e
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr -> ""
  | MatrixAccess(v, e1, e2) -> v ^ "[" ^ string_of_expr e1 ^ ", " ^ string_of_expr e2 ^ "]"
  | ImageRedAccess(v) -> v ^ ".red"
  | ImageGreenAccess(v) -> v ^ ".green"
  | ImageBlueAccess(v) -> v ^ ".blue"
  | ImageGrayscaleAccess(v) -> v ^ ".grayscale"
  | MatrixRows(v) -> v ^ ".rows"
  | MatrixCols(v) -> v ^ ".cols"

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ "; " ^ string_of_expr e2 ^ "; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
  | Variable(t, v, e) -> if string_of_expr e = ""
      then string_of_typ t ^ " " ^ v ^ ";\n" (* No value assigned *)
      else string_of_typ t ^ " " ^ string_of_expr e ^ ";\n" (* w/ value assigned *)
  | MatrixAssign(t, v, e) -> "matrix::" ^ string_of_typ t ^ " " ^ v ^ " = " ^ string_of_expr e ^ ";\n"
  | MatrixAccessAssign(v, e1, e2, e3) -> v ^ "[" ^ string_of_expr e1 ^ ", " ^ string_of_expr e2 ^ "] = " ^ string_of_expr e3 ^ ";"

let string_of_vdecl (t, id, e) = if string_of_expr e = ""
  then string_of_typ t ^ " " ^ id ^ ";\n"
  else string_of_typ t ^ " " ^ string_of_expr e ^ ";\n"

let string_of_fdecl fdecl =
  "fun::" ^ string_of_typ fdecl.typ ^ " " ^ fdecl.fname ^ "(" ^
    String.concat ", " (List.map string_of_vdecl fdecl.formals) ^
  ")\n{\n" ^ String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_decl = function
  | StmtDecl(s) -> string_of_stmt s
  | FuncDecl(f) -> string_of_fdecl f

let string_of_program decls =
  String.concat "" (List.map string_of_decl decls) ^ "\n"