(* Semantic checking for the Lucifer compiler
Authors: Elliott Morelli, Cherry Chu, Michael Fagan
Citation: MicroC semant.ml *)

open Ast
open Sast

module StringMap = Map.Make(String)

(* Semantic checking of the AST. Returns an SAST if successful,
   throws an exception if something is wrong.

   Check each global variable, then check each function *)
let check (globals, functions) =

  (* Verify a list of bindings has no void types or duplicate names *)
  let check_binds (kind : string) (binds : bind list) =
    List.iter (function
	(Void, b) -> raise (Failure ("illegal void " ^ kind ^ " " ^ b))
      | _ -> ()) binds;
    let rec dups = function
        [] -> ()
      |	((_,n1) :: (_,n2) :: _) when n1 = n2 ->
	  raise (Failure ("duplicate " ^ kind ^ " " ^ n1))
      | _ :: t -> dups t
    in dups (List.sort (fun (_,a) (_,b) -> compare a b) binds)
  in

  (**** Check global variables ****)

  check_binds "global" globals;

  (**** Check functions ****)

  (* Collect function declarations for built-in functions: no bodies *)
  let built_in_decls = 
    StringMap.add "print" { typ = Void; fname = "print"; formals = [(Int, "x")]; locals = []; body = [] }
    (StringMap.add "printf" { typ = Void; fname = "printf"; formals = [(Float, "x")]; locals = []; body = [] }
    (StringMap.add "printb" { typ = Void; fname = "printb"; formals = [(Bool, "x")]; locals = []; body = [] }
    (StringMap.add "prints" { typ = Void; fname = "prints"; formals = [(String, "x")]; locals = []; body = [] }
    (StringMap.add "initsdl" { typ = Void; fname = "initsdl"; formals = []; locals = []; body = [] }
    (StringMap.add "Player" { typ = Void; fname = "Player"; formals = [(Int, "n"); (Int, "hy"); (Int, "hx"); (String, "texture"); (Int, "y"); (Int, "x")]; locals = []; body = [] }
    (StringMap.add "Entity" { typ = Void; fname = "Entity"; formals = [(Int, "hy"); (Int, "hx"); (String, "texture"); (Int, "y"); (Int, "x")];
 locals = []; body = [] }
    (StringMap.add "isKeyPressed" { typ = Bool; fname = "isKeyPressed"; formals = [(Int, "x")]; locals = []; body = [] }
    (StringMap.add "getPlayerX" { typ = Int; fname = "getPlayerX"; formals = [(Player, "p")]; locals = []; body = [] }
    (StringMap.add "getPlayerY" { typ = Int; fname = "getPlayerY"; formals = [(Player, "p")]; locals = []; body = [] }
    (StringMap.add "setPlayerX" { typ = Void; fname = "setPlayerX"; formals = [(Player, "p");(Int, "x")]; locals = []; body = [] }
    (StringMap.add "setPlayerY" { typ = Void; fname = "setPlayerY"; formals = [(Player, "p");(Int, "x")]; locals = []; body = [] }
    (StringMap.add "getPlayerControl" { typ = Int; fname = "getPlayerControl"; formals = [(Player, "p"); (Int, "index")]; locals = []; body = [] }
    (StringMap.add "zeroPlayerControls" { typ = Void; fname = "zeroPlayerControls"; formals = [(Player, "p")]; locals = []; body = [] }
    (StringMap.add "getPlayerText" { typ = String; fname = "getPlayerText"; formals = [(Player, "e")]; locals = []; body = [] }
    (StringMap.add "getPlayerHx" { typ = Int; fname = "getPlayerHx"; formals = [(Player, "e")]; locals = []; body = [] }
    (StringMap.add "getPlayerHy" { typ = Int; fname = "getPlayerHy"; formals = [(Player, "e")]; locals = []; body = [] }
    (StringMap.add "addPlayerHitBox" { typ = Void; fname = "addPlayerHitBox"; formals = [(Player, "p"); (Int, "x"); (Int, "y")]; locals = []; body = [] }
    (StringMap.add "changePlayerX" { typ = Void; fname = "changePlayerX"; formals = [(Player, "p"); (Int, "x")]; locals = []; body = [] }
    (StringMap.add "changePlayerY" { typ = Void; fname = "changePlayerY"; formals = [(Player, "p"); (Int, "y")]; locals = []; body = [] }
    (StringMap.add "controlPlayer" { typ = Void; fname = "controlPlayer"; formals = [(Player, "p"); (Int, "y")]; locals = []; body = [] }
    (StringMap.add "addPlayerControl" { typ = Void; fname = "addPlayerControl"; formals = [(Player, "p"); (Int, "index"); (Int, "code")]; locals = []; body = [] }
    (StringMap.add "getEntityX" { typ = Int; fname = "getEntityX"; formals = [(Entity, "e")]; locals = []; body = [] }
    (StringMap.add "getEntityY" { typ = Int; fname = "getEntityY"; formals = [(Entity, "e")]; locals = []; body = [] }
    (StringMap.add "getEntityText" { typ = String; fname = "getEntityText"; formals = [(Entity, "e")]; locals = []; body = [] }
    (StringMap.add "getEntityHx" { typ = Int; fname = "getEntityHx"; formals = [(Entity, "e")]; locals = []; body = [] }
    (StringMap.add "getEntityHy" { typ = Int; fname = "getEntityHy"; formals = [(Entity, "e")]; locals = []; body = [] }
    (StringMap.add "setEntityX" { typ = Void; fname = "setEntityX"; formals = [(Entity, "e"); (Int, "n")]; locals = []; body = [] }
    (StringMap.add "setEntityY" { typ = Void; fname = "setEntityY"; formals = [(Entity, "e"); (Int, "n")]; locals = []; body = [] }
    (StringMap.add "addEntityHitBox" { typ = Void; fname = "addEntityHitBox"; formals = [(Entity, "e"); (Int, "x"); (Int, "y")]; locals = []; body = [] }
    (StringMap.add "changeEntityX" { typ = Void; fname = "changeEntityX"; formals = [(Entity, "e"); (Int, "x")]; locals = []; body = [] }
    (StringMap.singleton "changeEntityY" { typ = Void; fname = "changeEntityY"; formals = [(Entity, "e"); (Int, "y")]; locals = []; body = [] }
    )))))))))))))))))))))))))))))))
  in

  (* Add function name to symbol table *)
  let add_func map fd = 
    let built_in_err = "function " ^ fd.fname ^ " may not be defined"
    and dup_err = "duplicate function " ^ fd.fname
    and make_err er = raise (Failure er)
    and n = fd.fname (* Name of the function *)
    in match fd with (* No duplicate functions or redefinitions of built-ins *)
         _ when StringMap.mem n built_in_decls -> make_err built_in_err
       | _ when StringMap.mem n map -> make_err dup_err
       | _ ->  StringMap.add n fd map 
  in
  (* Collect all function names into one symbol table *)
  let function_decls = List.fold_left add_func built_in_decls functions
  in
  
  (* Return a function from our symbol table *)
  let find_func s = 
    try StringMap.find s function_decls
    with Not_found -> raise (Failure ("unrecognized function " ^ s))
  in

  let _ = find_func "main" in (* Ensure "main" is defined *)

  let check_function func =
    (* Make sure no formals or locals are void or duplicates *)
    check_binds "formal" func.formals;
    check_binds "local" func.locals;

    (* Raise an exception if the given rvalue type cannot be assigned to
       the given lvalue type *)
    let check_assign lvaluet rvaluet err =
       if lvaluet = rvaluet then lvaluet else raise (Failure err)
    in   

    (* Build local symbol table of variables for this function *)
    let symbols = List.fold_left (fun m (ty, name) -> StringMap.add name ty m)
	                StringMap.empty (globals @ func.formals @ func.locals )
    in
    
    (*Build hash table of global and function local Entity and Player declarations*)
    let obj_hash = Hashtbl.create 123456 in

    let obj_decls = List.iter (fun (ty,name) -> match ty with 
                                    Player -> Hashtbl.add  obj_hash name ty
                                    | Entity -> Hashtbl.add obj_hash name ty 
                                    |_ -> ()) (globals @ func.locals); in

    (* Return a variable from our local symbol table *)
    let type_of_identifier s =
      try StringMap.find s symbols
      with Not_found -> raise (Failure ("undeclared identifier " ^ s))
    in

    (*Function to remove a declaration from the hash once it is instantiated*)
    let obj_instantiated s =
      try Hashtbl.remove obj_hash s
      with Not_found -> ()
       in

    (* Return a semantically-checked expression, i.e., with a type *)
    let rec expr = function
        Literal  l -> (Int, SLiteral l)
      | Fliteral l -> (Float, SFliteral l)
      | BoolLit l  -> (Bool, SBoolLit l)
      | CLit l     -> (String, SCLit l)
      | Noexpr     -> (Void, SNoexpr)
      | Id s       -> (type_of_identifier s, SId s)
      | Assign(var, e) as ex -> 
          let lt = type_of_identifier var
          and (rt, e') = expr e in 
          (*remove e from obj_hash as it is instantiated to check obj instantiation**)
          let err = "illegal assignment " ^ string_of_typ lt ^ " = " ^ 
            string_of_typ rt ^ " in " ^ string_of_expr ex
          in ignore(obj_decls = obj_instantiated var); (check_assign lt rt err, SAssign(var, (rt, e')))
      | Unop(op, e) as ex -> 
          let (t, e') = expr e in
          let ty = match op with
            Neg when t = Int || t = Float -> t
          | Not when t = Bool -> Bool
          | _ -> raise (Failure ("illegal unary operator " ^ 
                                 string_of_uop op ^ string_of_typ t ^
                                 " in " ^ string_of_expr ex))
          in (ty, SUnop(op, (t, e')))
      | Binop(e1, op, e2) as e -> 
          let (t1, e1') = expr e1 
          and (t2, e2') = expr e2 in
          (* All binary operators require operands of the same type *)
          let same = t1 = t2 in
          (* Determine expression type based on operator and operand types *)
          let ty = match op with
            Add | Sub | Mult | Div | Mod when same && t1 = Int   -> Int
          | Add | Sub | Mult | Div when same && t1 = Float -> Float
          | Equal | Neq            when same               -> Bool
          | Less | Leq | Greater | Geq
                     when same && (t1 = Int || t1 = Float) -> Bool
          | And | Or when same && t1 = Bool -> Bool
          | _ -> raise (
	      Failure ("illegal binary operator " ^
                       string_of_typ t1 ^ " " ^ string_of_op op ^ " " ^
                       string_of_typ t2 ^ " in " ^ string_of_expr e))
          in (ty, SBinop((t1, e1'), op, (t2, e2')))
      | Call(fname, args) as call -> 
          let fd = find_func fname in
          let param_length = List.length fd.formals in
          if List.length args != param_length then
            raise (Failure ("expecting " ^ string_of_int param_length ^ 
                            " arguments in " ^ string_of_expr call))
          else let check_call (ft, _) e = 
            let (et, e') = expr e in 
            let err = "illegal argument found " ^ string_of_typ et ^
              " expected " ^ string_of_typ ft ^ " in " ^ string_of_expr e
            in (check_assign ft et err, e')
          in 
          let args' = List.map2 check_call fd.formals args
          in (fd.typ, SCall(fname, args'))

      | NewPlayer(e1, e2, e3, e4) ->
        let (t1, e1') = expr e1 in
        let (t2, e2') = expr e2 in
        let (t3, e3') = expr e3 in
        let (t4, e4') = expr e4 in
        if t1 != Int then raise (Failure ("expecting Int type for x position argument"))
        else if t2 != Int then raise (Failure ("expecting Int type for y position argument"))
        else if t3 != String then raise (Failure ("expecting String type for texture argument"))
        else if t4 != Int then raise (Failure ("expecting Int type for control array size argument"))
        else (match e4 with
                 Literal n -> if n < 4 then raise (Failure ("expecting a minimum size 4(up,down,left,right) for control array"))
                              else (Player, SNewPlayer((t1, e1'), (t2, e2'), (t3, e3'), (t4, e4')))
               | _ -> raise (Failure ("expecting a minimum size 4(up,down,left,right) for control array"))
             )
      | NewEntity(e1, e2, e3) ->
        let (t1, e1') = expr e1 in
        let (t2, e2') = expr e2 in
        let (t3, e3') = expr e3 in
        if t1 != Int then raise (Failure ("expecting Int type for x position argument"))
        else if t2 != Int then raise (Failure ("expecting Int type for y position argument"))
        else if t3 != String then raise (Failure ("expecting String type for texture argument"))
        else (Entity, SNewEntity((t1, e1'), (t2, e2'), (t3, e3')))
    in

    let check_bool_expr e = 
      let (t', e') = expr e
      and err = "expected Boolean expression in " ^ string_of_expr e
      in if t' != Bool then raise (Failure err) else (t', e') 
    in

    (*checking that objs are instantiated as they all would be removed*)
    let check_decls h = match Hashtbl.length h with
      0 -> ()
      |_ -> raise (Failure ("all entities and players must be instantiated")) 
      
    in

    (* Return a semantically-checked statement i.e. containing sexprs *)
    let rec check_stmt = function
        Expr e -> SExpr (expr e)
      | If(p, b1, b2) -> SIf(check_bool_expr p, check_stmt b1, check_stmt b2)
      | For(e1, e2, e3, st) ->
	  SFor(expr e1, check_bool_expr e2, expr e3, check_stmt st)
      | While(p, s) -> SWhile(check_bool_expr p, check_stmt s)
      | RunGame(e1, e2, s) -> SRunGame(check_bool_expr e1, expr e2, check_stmt s)
      | Return e -> let (t, e') = expr e in
        if t = func.typ then SReturn (t, e') 
        else raise (
	  Failure ("return gives " ^ string_of_typ t ^ " expected " ^
		   string_of_typ func.typ ^ " in " ^ string_of_expr e))
	    
	    (* A block is correct if each statement is correct and nothing
	       follows any Return statement.  Nested blocks are flattened. *)
      | Block sl -> 
          let rec check_stmt_list = function
              [Return _ as s] -> [check_stmt s]
            | Return _ :: _   -> raise (Failure "nothing may follow a return")
            | Block sl :: ss  -> check_stmt_list (sl @ ss) (* Flatten blocks *)
            | s :: ss         -> check_stmt s :: check_stmt_list ss
            | []              -> []
          in SBlock(check_stmt_list sl)

    in (* body of check_function, checks for obj instantiation on 258 *)
    {
      
      styp = func.typ;
      sfname = func.fname;
      sformals = func.formals;
      slocals  = func.locals;
      sbody = match check_stmt (Block func.body) with
	SBlock(sl) -> check_decls obj_hash ; sl
      | _ -> raise (Failure ("internal error: block didn't become a block?"));   
    };
  in (globals, List.map check_function functions)
