(* Code generation for Lucifer: 
Authors: Elliott Morelli, Cherry Chu, Michael Fagan
Citation: MicroC codegen.ml
*)

module L = Llvm
module A = Ast
open Sast 

module StringMap = Map.Make(String)

(* translate : Sast.program -> Llvm.module *)
let translate (globals, functions) =
  let context    = L.global_context () in
  
  (* Create the LLVM compilation module into which
     we will generate code *)
  let the_module = L.create_module context "Lucifer" in

  (* Get types from the context *)
  let i32_t      = L.i32_type    context
  and i8_t       = L.i8_type     context
  and i1_t       = L.i1_type     context
  and float_t    = L.double_type context
  and void_t     = L.void_type   context
  in

  let player_t = L.named_struct_type context "player_t" in
  L.struct_set_body player_t [|i32_t; i32_t; L.pointer_type i8_t; i32_t; i32_t; L.pointer_type i32_t;  L.pointer_type i8_t;|] false;

  let entity_t = L.named_struct_type context "entity_t" in
  L.struct_set_body entity_t [|i32_t; i32_t; L.pointer_type i8_t; L.pointer_type i8_t; i32_t; i32_t|] false;

  (* Return the LLVM type for a Lucifer type *)
  let ltype_of_typ = function
      A.Int   -> i32_t
    | A.Bool  -> i1_t
    | A.Float -> float_t
    | A.String-> L.pointer_type i8_t
    | A.Void  -> void_t
    | A.Player -> L.pointer_type player_t
    | A.Entity -> L.pointer_type entity_t
  in

  (* Create a map of global variables after creating each *)
  let global_vars : L.llvalue StringMap.t =
    let global_var m (t, n) = 
      let init = match t with
          A.Float -> L.const_float (ltype_of_typ t) 0.0
        | A.Entity -> print_string("in global") ; L.const_int (ltype_of_typ t) 99
        | _ -> L.const_int (ltype_of_typ t) 0
      in StringMap.add n (L.define_global n init the_module) m in
    List.fold_left global_var StringMap.empty globals in

  let printf_t : L.lltype = 
      L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func : L.llvalue = 
      L.declare_function "printf" printf_t the_module in

  let initsdl_t : L.lltype =
      L.var_arg_function_type i32_t [| i32_t |] in
  let initsdl_func : L.llvalue =
      L.declare_function "initsdl" initsdl_t the_module in

  let initentity_t : L.lltype = 
      L.var_arg_function_type i32_t [|L.pointer_type entity_t|] in
  let initentity_func : L.llvalue =
      L.declare_function "initEntity" initentity_t the_module in
   
  let initplayer_t : L.lltype = 
      L.var_arg_function_type i32_t [|L.pointer_type player_t|] in
  let initplayer_func : L.llvalue =
      L.declare_function "initPlayer" initplayer_t the_module in

  (*sdl functions used in runGame() that must be called before and after queueing entity and player textures *)

  let preparescene_t : L.lltype =
        L.var_arg_function_type i32_t [| i32_t |] in
  let preparescene_func : L.llvalue =
        L.declare_function "prepareScene" preparescene_t the_module in
  
  let presentscene_t : L.lltype =
            L.var_arg_function_type i32_t [| i32_t |] in
  let presentscene_func : L.llvalue =
            L.declare_function "presentScene" presentscene_t the_module in


  let iskeypressed_t : L.lltype =
            L.var_arg_function_type i1_t [| i32_t |] in
  let iskeypressed_func : L.llvalue =
            L.declare_function "isKeyPressed" iskeypressed_t the_module in
    

  let update_e_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t |] in
  let update_e_func : L.llvalue =
      L.declare_function "updateE" update_e_t the_module in

  let update_p_t : L.lltype =
        L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let update_p_func : L.llvalue =
        L.declare_function "updateP" update_p_t the_module in
  

  let getplayerx_t : L.lltype = 
      L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let getplayerx_func : L.llvalue  = 
      L.declare_function "getPlayerX" getplayerx_t the_module in
  
  let getplayery_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let getplayery_func : L.llvalue  =
      L.declare_function "getPlayerY" getplayery_t the_module in

  let setplayerx_t : L.lltype = 
        L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
  let setplayerx_func : L.llvalue  = 
        L.declare_function "setPlayerX" setplayerx_t the_module in
    
  let setplayery_t : L.lltype =
        L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
   let setplayery_func : L.llvalue  =
        L.declare_function "setPlayerY" setplayery_t the_module in

  let getplayercontrol_t : L.lltype =
          L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
  let getplayercontrol_func : L.llvalue  =
      L.declare_function "getPlayerControl" getplayercontrol_t the_module in

  let zeroplayercontrols_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let zeroplayercontrols_func : L.llvalue  =
      L.declare_function "zeroPlayerControls" zeroplayercontrols_t the_module in

  let getplayertext_t : L.lltype =
      L.var_arg_function_type (L.pointer_type i32_t) [| L.pointer_type player_t |] in
  let getplayertext_func : L.llvalue  =
      L.declare_function "getPlayerText" getplayertext_t the_module in

  let getplayerhx_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let getplayerhx_func : L.llvalue  =
      L.declare_function "getPlayerHx" getplayerhx_t the_module in

  let getplayerhy_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t |] in
  let getplayerhy_func : L.llvalue  =
      L.declare_function "getPlayerHy" getplayerhy_t the_module in

  let addplayerhitbox_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t; i32_t |] in
  let addplayerhitbox_func : L.llvalue  =
      L.declare_function "addPlayerHitBox" addplayerhitbox_t the_module in

  let changeplayerx_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
  let changeplayerx_func : L.llvalue  =
      L.declare_function "changePlayerX" changeplayerx_t the_module in

  let changeplayery_t : L.lltype =
          L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
  let changeplayery_func : L.llvalue  =
      L.declare_function "changePlayerY" changeplayery_t the_module in

  let addplayercontrol_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t; i32_t |] in
  let addplayercontrol_func : L.llvalue  =
      L.declare_function "addPlayerControl" addplayercontrol_t the_module in

  let controlplayer_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type player_t; i32_t |] in
  let controlplayer_func : L.llvalue  =
      L.declare_function "controlPlayer" controlplayer_t the_module in

  let getentityx_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t |] in
  let getentityx_func : L.llvalue  =
      L.declare_function "getEntityX" getentityx_t the_module in

  let getentity_y_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t |] in
  let getentity_y_func : L.llvalue  =
      L.declare_function "getEntityY" getentity_y_t the_module in

  let getentitytext_t : L.lltype =
      L.var_arg_function_type (L.pointer_type i32_t) [| L.pointer_type entity_t |] in
  let getentitytext_func : L.llvalue  =
      L.declare_function "getEntityText" getentitytext_t the_module in

  let getentityhx_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t |] in
  let getentityhx_func : L.llvalue  =
      L.declare_function "getEntityHx" getentityhx_t the_module in

  let getentityhy_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t |] in
  let getentityhy_func : L.llvalue  =
      L.declare_function "getEntityHy" getentityhy_t the_module in
  
  let setentityx_t : L.lltype =
          L.var_arg_function_type i32_t [| L.pointer_type entity_t; i32_t |] in
  let setentityx_func : L.llvalue  =
      L.declare_function "setEntityX" setentityx_t the_module in

  let setentity_y_t : L.lltype =
        L.var_arg_function_type i32_t [| L.pointer_type entity_t; i32_t |] in
  let setentity_y_func : L.llvalue  =
    L.declare_function "setEntityY" setentity_y_t the_module in

  let addentityhitbox_t : L.lltype =
      L.var_arg_function_type i32_t [| L.pointer_type entity_t; i32_t; i32_t |] in
  let addentityhitbox_func : L.llvalue  =
      L.declare_function "addEntityHitBox" addentityhitbox_t the_module in

  let change_entityx_t : L.lltype =
          L.var_arg_function_type i32_t [| L.pointer_type entity_t; i32_t |] in
  let change_entityx_func : L.llvalue  =
      L.declare_function "changeEntityX" change_entityx_t the_module in

  let change_entity_y_t : L.lltype =
          L.var_arg_function_type i32_t [| L.pointer_type entity_t; i32_t |] in
  let change_entity_y_func : L.llvalue  =
      L.declare_function "changeEntityY" change_entity_y_t the_module in

  (* Define each function (arguments and return type) so we can 
     call it even before we've created its body *)
  let function_decls : (L.llvalue * sfunc_decl) StringMap.t =
    let function_decl m fdecl =
      let name = fdecl.sfname
      and formal_types = 
	Array.of_list (List.map (fun (t,_) -> ltype_of_typ t) fdecl.sformals)
      in let ftype = L.function_type (match fdecl.sfname with
          "main" -> i32_t
        | _ -> ltype_of_typ fdecl.styp)
      formal_types
      in
      StringMap.add name (L.define_function name ftype the_module, fdecl) m in
    List.fold_left function_decl StringMap.empty functions in
  
  (* Fill in the body of the given function *)
  let build_function_body fdecl =
    let (the_function, _) = StringMap.find fdecl.sfname function_decls in
    let builder = L.builder_at_end context (L.entry_block the_function) in

    let int_format_str = L.build_global_stringptr "%d\n" "fmt" builder
    and str_format_str = L.build_global_stringptr "%s\n" "fmt" builder
    and float_format_str = L.build_global_stringptr "%g\n" "fmt" builder in

    (* Construct the function's "locals": formal arguments and locally
       declared variables.  Allocate each on the stack, initialize their
       value, if appropriate, and remember their values in the "locals" map *)
    let local_vars =
      let add_formal m (t, n) p = 
        L.set_value_name n p;
	let local = L.build_alloca (ltype_of_typ t) n builder in
        ignore (L.build_store p local builder);
	StringMap.add n local m 

      (* Allocate space for any locally declared variables and add the
       * resulting registers to our map *)
      and add_local m (t, n) = 
	let local_var = L.build_alloca (ltype_of_typ t) n builder
	in 
    StringMap.add n local_var m 
      in

      let formals = List.fold_left2 add_formal StringMap.empty fdecl.sformals
          (Array.to_list (L.params the_function)) in
      List.fold_left add_local formals fdecl.slocals 
    in

    (*creates list of entity and player declarations as expressions*)
        let object_vars = 
          let add_local_obj l (t,n) =
            match t with 
            | A.Entity -> (t, (SId n)) :: l
            | A.Player -> (t, (SId n)) :: l
            | _ -> l
            in 
            List.fold_left add_local_obj [] fdecl.slocals
            in 

    (* Return the value for a variable or formal argument.
       Check local names first, then global names *)
    let lookup n = try StringMap.find n local_vars
                   with Not_found -> try StringMap.find n global_vars
                   with Not_found -> raise (Failure "internal error: variable may not be initialized")
    in

    (* Construct code for an expression; return its value *)
    let rec expr builder ((_, e) : sexpr) = match e with
	SLiteral i  -> L.const_int i32_t i
      | SBoolLit b  -> L.const_int i1_t (if b then 1 else 0)
      | SFliteral l -> L.const_float_of_string float_t l
      | SCLit s     -> L.build_global_stringptr	s "str" builder
      | SNoexpr     -> L.const_int i32_t 0
      | SId s       -> L.build_load (lookup s) s builder
      | SAssign (s, e) -> let e' = expr builder e in
                          ignore(L.build_store e' (lookup s) builder); e'
      | SBinop ((A.Float,_ ) as e1, op, e2) ->
	  let e1' = expr builder e1
	  and e2' = expr builder e2 in
	  (match op with 
	    A.Add     -> L.build_fadd
	  | A.Sub     -> L.build_fsub
	  | A.Mult    -> L.build_fmul
	  | A.Div     -> L.build_fdiv 
          | A.Mod     -> 
              raise (Failure "internal error: semant should have rejected mod on flaot")
	  | A.Equal   -> L.build_fcmp L.Fcmp.Oeq
	  | A.Neq     -> L.build_fcmp L.Fcmp.One
	  | A.Less    -> L.build_fcmp L.Fcmp.Olt
	  | A.Leq     -> L.build_fcmp L.Fcmp.Ole
	  | A.Greater -> L.build_fcmp L.Fcmp.Ogt
	  | A.Geq     -> L.build_fcmp L.Fcmp.Oge
	  | A.And | A.Or ->
	      raise (Failure "internal error: semant should have rejected and/or on float")
	  ) e1' e2' "tmp" builder
      | SBinop (e1, op, e2) ->
	  let e1' = expr builder e1
	  and e2' = expr builder e2 in
	  (match op with
	    A.Add     -> L.build_add
	  | A.Sub     -> L.build_sub
	  | A.Mult    -> L.build_mul
          | A.Div     -> L.build_sdiv
          | A.Mod     -> L.build_srem
	  | A.And     -> L.build_and
	  | A.Or      -> L.build_or
	  | A.Equal   -> L.build_icmp L.Icmp.Eq
	  | A.Neq     -> L.build_icmp L.Icmp.Ne
	  | A.Less    -> L.build_icmp L.Icmp.Slt
	  | A.Leq     -> L.build_icmp L.Icmp.Sle
	  | A.Greater -> L.build_icmp L.Icmp.Sgt
	  | A.Geq     -> L.build_icmp L.Icmp.Sge
	  ) e1' e2' "tmp" builder
      | SUnop(op, ((t, _) as e)) ->
          let e' = expr builder e in
	  (match op with
	    A.Neg when t = A.Float -> L.build_fneg 
	  | A.Neg                  -> L.build_neg
          | A.Not                  -> L.build_not) e' "tmp" builder
      | SCall ("print", [e]) | SCall ("printb", [e]) ->
	  L.build_call printf_func [| int_format_str ; (expr builder e) |]
	    "printf" builder
      | SCall ("initsdl", []) ->
          L.build_call initsdl_func [| ((L.const_int i32_t 0)) |] "initsdl" builder
      | SCall ("printf", [e]) -> 
	  L.build_call printf_func [| float_format_str ; (expr builder e) |] "printf" builder
      | SCall ("prints", [e]) ->
          L.build_call printf_func [| str_format_str; (expr builder e) |] "printf" builder
      | SCall ("getPlayerX", [e]) ->
          L.build_call getplayerx_func [| (expr builder e) |] "getPlayerX" builder
      | SCall ("getPlayerY", [e]) ->
          L.build_call getplayery_func [| (expr builder e) |] "getPlayerY" builder
      | SCall ("setPlayerX", [e1;e2]) ->
          L.build_call setplayerx_func [| (expr builder e1); (expr builder e2) |] "setPlayerX" builder
      | SCall ("setPlayerY", [e1;e2]) ->
          L.build_call setplayery_func [| (expr builder e1); (expr builder e2) |] "setPlayerY" builder
      | SCall ("getPlayerControl", [e1; e2]) ->
          L.build_call getplayercontrol_func [| (expr builder e1); (expr builder e2) |] "getPlayerControl" builder
      | SCall ("getPlayerText", [e]) ->
          L.build_call getplayertext_func [| (expr builder e) |] "getPlayerText" builder
      | SCall ("getPlayerHx", [e]) ->
          L.build_call getplayerhx_func [| (expr builder e) |] "getPlayerHx" builder
      | SCall ("getPlayerHy", [e]) ->
          L.build_call getplayerhy_func [| (expr builder e) |] "getPlayerHy" builder
      | SCall ("addPlayerHitBox", [e1; e2; e3]) ->
          L.build_call addplayerhitbox_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "addPlayerHitBox" builder
      | SCall ("changePlayerX", [e1; e2]) ->
          L.build_call changeplayerx_func [| (expr builder e1); (expr builder e2) |] "changePlayerX" builder
      | SCall ("changePlayerY", [e1; e2]) ->
          L.build_call changeplayery_func [| (expr builder e1); (expr builder e2) |] "changePlayerY" builder
      | SCall ("addPlayerControl", [e1; e2; e3]) ->
          L.build_call addplayercontrol_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "addPlayerControl" builder
      | SCall ("controlPlayer", [e1; e2]) ->
          L.build_call controlplayer_func [| (expr builder e1); (expr builder e2)|] "controlPlayer" builder
      | SCall ("getEntityX", [e]) ->
          L.build_call getentityx_func [| (expr builder e) |] "getEntityX" builder
      | SCall ("getEntityY", [e]) ->
          L.build_call getentity_y_func [| (expr builder e) |] "getEntityY" builder
      | SCall ("getEntityText", [e]) ->
          L.build_call getentitytext_func [| (expr builder e) |] "getEntityText" builder
      | SCall ("getEntityHx", [e]) ->
          L.build_call getentityhx_func [| (expr builder e) |] "getEntityHx" builder
      | SCall ("getEntityHy", [e]) ->
          L.build_call getentityhy_func [| (expr builder e) |] "getEntityHy" builder
      | SCall ("setEntityX", [e1; e2]) ->
          L.build_call setentityx_func [| (expr builder e1); (expr builder e2) |] "setEntityX" builder
      | SCall ("setEntityY", [e1; e2]) ->
          L.build_call setentity_y_func [| (expr builder e1); (expr builder e2) |] "setEntityY" builder
      | SCall ("addEntityHitBox", [e1; e2; e3]) ->
          L.build_call addentityhitbox_func [| (expr builder e1); (expr builder e2); (expr builder e3) |] "addEntityHitBox" builder
      | SCall ("changeEntityX", [e1; e2]) ->
          L.build_call change_entityx_func [| (expr builder e1); (expr builder e2) |] "changeEntityX" builder
      | SCall ("updateEntities",[]) -> 
      (*this function is called in runGame() to render entities and players at updated positions*)
          let update_entity (t,n) = match t with 
                A.Player -> L.build_call update_p_func [| (expr builder (t,n)) |] "updateP" builder
                | _ -> L.build_call update_e_func [| (expr builder (t,n)) |] "updateE" builder 
                in 
          let update_entities l = 
                    ignore(List.map update_entity l); update_entity (List.hd l) 
                    in    
            update_entities object_vars  
      | SCall ("initEntities",[]) -> 
      (*this function is called before runGame() iterates to load entity and player textures*)
          let init_entity (t,n) = match t with
                A.Player ->  L.build_call initplayer_func [| (expr builder (t,n)) |] "initPlayer" builder 
                |_ -> L.build_call initentity_func [| (expr builder (t,n)) |] "initEntity" builder
                in 
          let init_entities l = 
                    ignore(List.map init_entity l); init_entity (List.hd l) 
                    in    
            init_entities object_vars 
      | SCall ("prepareScene",[]) ->
          L.build_call preparescene_func [| L.const_int i32_t 0 |] "prepareScene" builder
      | SCall ("presentScene", [e])->
          L.build_call presentscene_func [| (expr builder e)|] "presentScene" builder   
      | SCall ("isKeyPressed", [e])->
          L.build_call iskeypressed_func [| (expr builder e)|] "isKeyPressed" builder 
      | SCall ("changeEntityY", [e1; e2]) ->
          L.build_call change_entity_y_func [| (expr builder e1); (expr builder e2) |] "changeEntityY" builder
      | SCall (f, args) ->
         let (fdef, fdecl) = StringMap.find f function_decls in
	     let llargs = List.rev (List.map (expr builder) (List.rev args)) in
	     let result = (match fdecl.styp with 
                        A.Void -> ""
                      | _ -> f ^ "_result") in
         L.build_call fdef (Array.of_list llargs) result builder
      | SNewPlayer(x, y, texture, n) ->
         let ptmp = L.build_alloca player_t "player_tmp" builder in
         let pptr = L.build_alloca (L.pointer_type player_t) "player_ptr" builder in
         let e1 = expr builder x
         and e2 = expr builder y
         and e3 = expr builder texture
         and e4 = expr builder (A.Int, SLiteral(0))
         and e5 = expr builder (A.Int, SLiteral(0))
         and e6 = expr builder n 
         and e7 = expr builder texture in
         let xtmp = L.build_struct_gep ptmp 0 "x" builder in
         ignore (L.build_store e1 xtmp builder);
         let ytmp = L.build_struct_gep ptmp 1 "y" builder in
         ignore (L.build_store e2 ytmp builder);
         let texturetmp = L.build_struct_gep ptmp 2 "texture" builder in
         ignore (L.build_store e3 texturetmp builder);
         let hxtmp = L.build_struct_gep ptmp 3 "hx" builder in
         ignore (L.build_store e4 hxtmp builder);
         let hytmp = L.build_struct_gep ptmp 4 "hy" builder in
         ignore (L.build_store e5 hytmp builder);
         
         let ntmp = L.build_struct_gep ptmp 5 "n" builder in

         let ltype = ltype_of_typ A.Int in
         let size_t = L.build_intcast (L.size_of ltype) i32_t "tmp" builder in
         let total_size = L.build_mul size_t e6 "tmp" builder in
         let total_size = L.build_add total_size (L.const_int i32_t 1) "tmp" builder in
         let arr_malloc = L.build_array_malloc ltype total_size "tmp" builder in
         let arr = L.build_pointercast arr_malloc (L.pointer_type ltype) "tmp" builder in

         ignore (L.build_store arr ntmp builder);

         ignore (L.build_store ptmp pptr builder);

         ignore (L.build_call zeroplayercontrols_func [| ptmp |] "zeroPlayerControls" builder);

         let texture2tmp = L.build_struct_gep ptmp 6 "text" builder in
         ignore (L.build_store e7 texture2tmp builder);

         L.build_load pptr "" builder

      | SNewEntity(x, y, texture) ->
         let etmp = L.build_alloca entity_t "entity_tmp" builder in
         let eptr = L.build_alloca (L.pointer_type entity_t) "entity_ptr" builder in
         let e1 = expr builder x
         and e2 = expr builder y
         and e3 = expr builder texture
         and e4 = expr builder texture
         and e5 = expr builder (A.Int, SLiteral(0))
         and e6 = expr builder (A.Int, SLiteral(0)) in
         let xtmp = L.build_struct_gep etmp 0 "x" builder in
         ignore (L.build_store e1 xtmp builder);
         let ytmp = L.build_struct_gep etmp 1 "y" builder in
         ignore (L.build_store e2 ytmp builder);
         let texturetmp = L.build_struct_gep etmp 2 "texture" builder in
         ignore (L.build_store e3 texturetmp builder);
         let texture2tmp = L.build_struct_gep etmp 3 "text" builder in
         ignore (L.build_store e4 texture2tmp builder);
         let hxtmp = L.build_struct_gep etmp 4 "hx" builder in
         ignore (L.build_store e5 hxtmp builder);
         let hytmp = L.build_struct_gep etmp 5 "hy" builder in
         ignore (L.build_store e6 hytmp builder);
         ignore (L.build_store etmp eptr builder);
         L.build_load eptr "" builder
    in
    
    (* LLVM insists each basic block end with exactly one "terminator" 
       instruction that transfers control.  This function runs "instr builder"
       if the current block does not already have a terminator.  Used,
       e.g., to handle the "fall off the end of the function" case. *)
    let add_terminal builder instr =
      match L.block_terminator (L.insertion_block builder) with
	Some _ -> ()
      | None -> ignore (instr builder) in
	
    (* Build the code for the given statement; return the builder for
       the statement's successor (i.e., the next instruction will be built
       after the one generated by this call) *)

    let rec stmt builder = function
	SBlock sl -> List.fold_left stmt builder sl
      | SExpr e -> ignore(expr builder e); builder 
      | SReturn e -> ignore(match fdecl.styp with
                              (* Special "return nothing" instr *)
                              A.Void -> (match fdecl.sfname with
                                    "main" -> L.build_ret (L.const_int i32_t 0) builder
                                  | _ -> L.build_ret_void builder
                                )
                              (* Build return statement *)
                            | _ -> L.build_ret (expr builder e) builder );
                     builder
      | SIf (predicate, then_stmt, else_stmt) ->
         let bool_val = expr builder predicate in
	 let merge_bb = L.append_block context "merge" the_function in
         let build_br_merge = L.build_br merge_bb in (* partial function *)

	 let then_bb = L.append_block context "then" the_function in
	 add_terminal (stmt (L.builder_at_end context then_bb) then_stmt)
	   build_br_merge;

	 let else_bb = L.append_block context "else" the_function in
	 add_terminal (stmt (L.builder_at_end context else_bb) else_stmt)
	   build_br_merge;

	 ignore(L.build_cond_br bool_val then_bb else_bb builder);
	 L.builder_at_end context merge_bb

      | SWhile (predicate, body) ->
	  let pred_bb = L.append_block context "while" the_function in
	  ignore(L.build_br pred_bb builder);

	  let body_bb = L.append_block context "while_body" the_function in
	  add_terminal (stmt (L.builder_at_end context body_bb) body)
	    (L.build_br pred_bb);

	  let pred_builder = L.builder_at_end context pred_bb in
	  let bool_val = expr pred_builder predicate in

	  let merge_bb = L.append_block context "merge" the_function in
	  ignore(L.build_cond_br bool_val body_bb merge_bb pred_builder);
	  L.builder_at_end context merge_bb

      (* Implement for loops as while loops *)
      | SFor (e1, e2, e3, body) -> stmt builder
	    ( SBlock [SExpr e1 ; SWhile (e2, SBlock [body ; SExpr e3]) ] )

    (*Implement runGame() as a while loop with calls to initialize textures and render objs*)
      | SRunGame (e1,e2, body) -> stmt builder 
      ( SBlock [SExpr (A.Void, SCall ("initEntities",[])); 
      SWhile (e1, SBlock [SExpr (A.Void, SCall ("prepareScene",[])); 
      SExpr (A.Void, SCall ("updateEntities",[])); 
      body; SExpr (A.Void, SCall ("presentScene",[e2]))]) ])
    in

    (* Build the code for each statement in the function *)
    let builder = stmt builder (SBlock fdecl.sbody) in

    (* Add a return if the last block falls off the end *)
    add_terminal builder (match fdecl.styp with
        A.Void -> (match fdecl.sfname with
              "main" -> L.build_ret (L.const_int i32_t 0)
            | _ -> L.build_ret_void
          )
      | A.Float -> L.build_ret (L.const_float float_t 0.0)
      | t -> L.build_ret (L.const_int (ltype_of_typ t) 0))
  in

  List.iter build_function_body functions;
  the_module
