/* Ocamlyacc parser for Lucifer 
Authors: Elliott Morelli, Cherry Chu, Michael Fagan
Citation: Microcparse.mly */

%{
open Ast
%}

%token SEMI LPAREN RPAREN LBRACE RBRACE COMMA PLUS MINUS TIMES DIVIDE ASSIGN MOD
%token NOT EQ NEQ LT LEQ GT GEQ AND OR
%token RETURN IF ELSE FOR WHILE RUNGAME INT BOOL FLOAT STRING VOID FUN
%token NEW PLAYER ENTITY DOT 
%token <int> LITERAL
%token <bool> BLIT
%token <string> ID FLIT CLIT SID
%token EOF

%start program
%type <Ast.program> program

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE MOD
%right NOT

%%

program:
  decls EOF { $1 }

decls:
   /* nothing */ { ([], [])               }
 | decls vdecl { (($2 :: fst $1), snd $1) }
 | decls fdecl { (fst $1, ($2 :: snd $1)) }

fdecl:
   FUN ID typ LPAREN formals_opt RPAREN LBRACE vdecl_list stmt_list RBRACE
     { { typ = $3;
	 fname = $2;
	 formals = List.rev $5;
	 locals = List.rev $8;
	 body = List.rev $9 } }


formals_opt:
    /* nothing */ { [] }
  | formal_list   { $1 }

formal_list:
    typ ID                   { [($1,$2)]     }
  | formal_list COMMA typ ID { ($3,$4) :: $1 }

typ:
    INT   { Int   }
  | BOOL  { Bool  }
  | FLOAT { Float }
  | STRING { String }
  | VOID   { Void   }
  | PLAYER { Player }
  | ENTITY { Entity }

vdecl_list:
    /* nothing */    { [] }
  | vdecl_list vdecl { $2 :: $1 }

vdecl:
   typ ID SEMI { ($1, $2) }

stmt_list:
    /* nothing */  { [] }
  | stmt_list stmt { $2 :: $1 }

b_stmt_list:
    /* nothing */  { [] }
  | b_stmt_list b_stmt { $2 :: $1 }

stmt:
    expr SEMI                               { Expr $1               }
  | RETURN expr_opt SEMI                    { Return $2             }
  | LBRACE stmt_list RBRACE                 { Block(List.rev $2)    }
  | IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt    { If($3, $5, $7)        }
  | FOR LPAREN expr_opt SEMI expr SEMI expr_opt RPAREN stmt
                                            { For($3, $5, $7, $9)   }
  | WHILE LPAREN expr RPAREN stmt           { While($3, $5)         }
  | RUNGAME LPAREN expr SEMI expr RPAREN b_stmt         { RunGame($3, $5, $7)       }

b_stmt:
    niexpr SEMI                               { Expr $1               }
  | RETURN niexpr_opt SEMI                    { Return $2             }
  | LBRACE b_stmt_list RBRACE                 { Block(List.rev $2)    }
  | IF LPAREN niexpr RPAREN b_stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN niexpr RPAREN b_stmt ELSE b_stmt    { If($3, $5, $7)        }
  | FOR LPAREN niexpr_opt SEMI niexpr SEMI niexpr_opt RPAREN b_stmt
                                            { For($3, $5, $7, $9)   }
  | WHILE LPAREN niexpr RPAREN b_stmt           { While($3, $5)         }
  | RUNGAME LPAREN niexpr SEMI niexpr RPAREN b_stmt         { RunGame($3, $5, $7)       }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

niexpr_opt:
    /* nothing */ { Noexpr }
  | niexpr          { $1 }

expr:
    LITERAL          { Literal($1)            }
  | FLIT	     { Fliteral($1)           }
  | BLIT             { BoolLit($1)            }
  | CLIT             { CLit($1)               }
  | ID               { Id($1)                 }
  | expr PLUS   expr { Binop($1, Add,     $3) }
  | expr MINUS  expr { Binop($1, Sub,     $3) }
  | expr TIMES  expr { Binop($1, Mult,    $3) }
  | expr DIVIDE expr { Binop($1, Div,     $3) }
  | expr EQ     expr { Binop($1, Equal,   $3) }
  | expr NEQ    expr { Binop($1, Neq,     $3) }
  | expr LT     expr { Binop($1, Less,    $3) }
  | expr LEQ    expr { Binop($1, Leq,     $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GEQ    expr { Binop($1, Geq,     $3) }
  | expr AND    expr { Binop($1, And,     $3) }
  | expr OR     expr { Binop($1, Or,      $3) }
  | expr MOD    expr { Binop($1, Mod,     $3) }
  | MINUS expr %prec NOT { Unop(Neg, $2)      }
  | NOT expr         { Unop(Not, $2)          }
  | ID ASSIGN expr   { Assign($1, $3)         }
  | ID LPAREN args_opt RPAREN { Call($1, $3)  }
  | LPAREN expr RPAREN { $2                   }
  | NEW PLAYER LPAREN expr COMMA expr COMMA expr COMMA expr RPAREN
                     { NewPlayer($4, $6, $8, $10) }
  | NEW ENTITY LPAREN expr COMMA expr COMMA expr RPAREN
                     { NewEntity($4, $6, $8)  }
  | ID DOT ID LPAREN args_opt RPAREN
                     { Call($3, List.rev (List.rev $5 @ [Id($1)])) }

niexpr:
    LITERAL          { Literal($1)            }
  | FLIT	     { Fliteral($1)           }
  | BLIT             { BoolLit($1)            }
  | CLIT             { CLit($1)               }
  | ID               { Id($1)                 }
  | niexpr PLUS   niexpr { Binop($1, Add,     $3) }
  | niexpr MINUS  niexpr { Binop($1, Sub,     $3) }
  | niexpr TIMES  niexpr { Binop($1, Mult,    $3) }
  | niexpr DIVIDE niexpr { Binop($1, Div,     $3) }
  | niexpr EQ     niexpr { Binop($1, Equal,   $3) }
  | niexpr NEQ    niexpr { Binop($1, Neq,     $3) }
  | niexpr LT     niexpr { Binop($1, Less,    $3) }
  | niexpr LEQ    niexpr { Binop($1, Leq,     $3) }
  | niexpr GT     niexpr { Binop($1, Greater, $3) }
  | niexpr GEQ    niexpr { Binop($1, Geq,     $3) }
  | niexpr AND    niexpr { Binop($1, And,     $3) }
  | niexpr OR     niexpr { Binop($1, Or,      $3) }
  | niexpr MOD    niexpr { Binop($1, Mod,     $3) }
  | MINUS niexpr %prec NOT { Unop(Neg, $2)      }
  | NOT niexpr         { Unop(Not, $2)          }
  | ID ASSIGN niexpr   { Assign($1, $3)         }
  | ID LPAREN nargs_opt RPAREN { Call($1, $3)  }
  | LPAREN niexpr RPAREN { $2                   }
  | ID DOT ID LPAREN nargs_opt RPAREN
                     { Call($3, List.rev (List.rev $5 @ [Id($1)])) }

args_opt:
    /* nothing */ { [] }
  | args_list  { List.rev $1 }

args_list:
    expr                    { [$1] }
  | args_list COMMA expr { $3 :: $1 }

nargs_opt:
    /* nothing */ { [] }
  | nargs_list  { List.rev $1 }

nargs_list:
    niexpr                    { [$1] }
  | nargs_list COMMA niexpr { $3 :: $1 }
