open Ast
open Sast

module StringMap = Map.Make(String)

type class_property_map = {
	field_map : Ast.typ StringMap.t;
	method_map : Ast.func_decl StringMap.t;
}

type env = {
	env_global_map : Ast.typ StringMap.t;
	env_local_map : Ast.typ StringMap.t;
	env_param_map : Ast.typ StringMap.t;
	env_function_map : Ast.func_decl StringMap.t;
	env_class_map : class_property_map StringMap.t;
	env_return_type : Ast.typ;
}


let define_builtin_functions =
	[
	   {typ = Void; fname = "cast"; formals = [(String, "x")];
       locals = []; body = []};
       {typ = Void; fname = "malloc"; formals = [(String, "x")];
       locals = []; body = []};
       {typ = Void; fname = "sizeof"; formals = [(String, "x")];
       locals = []; body = []};
		{typ = Void; fname = "print"; formals = [(Any, "x")];
       locals = []; body = [] };
	   {typ = String; fname = "toString"; formals = [(Any,"x")];
	   locals = []; body = [] };
	    
	]

let build_maps(functions,builtins,globals) =
(*TODO: check duplicated function declarations here*)
	let function_map = List.fold_left (fun m fd -> 
		if (StringMap.mem fd.fname m) then
			raise (Failure("duplicated function declaration " ^ fd.fname))
		else
			StringMap.add fd.fname fd m)
                         StringMap.empty (functions @ builtins)  in 
	let global_map = List.fold_left (fun m (t,n) -> 
	if (StringMap.mem n m) then
			raise (Failure("duplicated global declaration " ^ n))
	else if (t = Void) then
			raise(Failure("global " ^ n ^ " shouldn't be void type"))
		else
			StringMap.add n t m)
                         StringMap.empty (globals)  in
						 (function_map,global_map)

let build_class_field_map(fields,globals) =
	List.fold_left (fun m (t,n) -> 
	if (StringMap.mem n m) then
			raise (Failure("duplicated declaration " ^ n ^ ",field " ^ string_of_typ(t) ^ " " ^ n ^ " already exists"))	
	else if (t = Void) then
			raise(Failure(n ^ " shouldn't be void type "))
		else
			StringMap.add n t m)
                         StringMap.empty (fields)

let build_class_method_map(cname,methods,functions) =
	let find_constructor = (fun f -> match f.typ with Constructortyp -> true  | _ -> false) in
	let get_constructor  = 
		let constructors = (List.find_all find_constructor methods) in
		let count = List.length constructors in 
		if List.length constructors < 1 then 
			raise (Failure("Missing constructor for class " ^ cname))
		else List.hd constructors 
	in
	ignore(get_constructor);
	List.fold_left (fun m fdecl -> 
		if (StringMap.mem fdecl.fname m) then
			raise (Failure("duplicated method declaration " ^ fdecl.fname))
		else
			StringMap.add fdecl.fname fdecl m)
                         StringMap.empty (methods)

let build_class_map(globals,functions,classes) =
	let class_map = List.fold_left (fun m cdecl ->
    	if (StringMap.mem cdecl.cname m) then
    		raise (Failure("duplicated class declaration" ^ cdecl.cname))
    	else
    		StringMap.add cdecl.cname 
    		{
    			field_map = build_class_field_map(cdecl.cbody.fields,globals);
    			method_map = build_class_method_map(cdecl.cname,cdecl.cbody.constructors@cdecl.cbody.methods,functions);
    		} 
    		m)
    		StringMap.empty (classes) in
    					 class_map

(* locals could overshadow params, params could overshadow globals *)
let get_id_type(env,s) = 
	try StringMap.find s env.env_local_map
	with | Not_found -> 
		try StringMap.find s env.env_param_map
		with |Not_found -> 
			try StringMap.find s env.env_global_map
		with |Not_found -> raise (Failure("undefined_id " ^ s))

let get_function_decl(env,fname) =
	try StringMap.find fname env.env_function_map
	with | Not_found -> raise (Failure("UndefinedFunction " ^ fname))

let get_sexpr_type(sexpr)= match (sexpr) with
    S_Literal(_) -> Int
  | S_BoolLit(_) -> Bool 
  | S_StringLit(_) -> String
  | S_FloatLit(_) -> Float
  | S_Id(_,t) -> t
  | S_Binop(_,_,_,t) -> t
  | S_Unop(_,_,t) -> t
  | S_Assign(_,_,t) -> t
  | S_Call(_,_,t) -> t
  | S_Noexpr -> Void
  | S_Null -> Null
  | S_Cast(_,_,t) -> t
  | S_ObjCreate(_,_,t) -> t
  | S_ObjAccess(_,_,t) -> t
  | S_ArrayCreate(t,_) -> Arraytype(t)
  | S_ArrayAccess(_,_,t) -> t
  | S_Delete(_) -> Void

let report_duplicate exceptf list =
let rec helper = function
n1 :: n2 :: _ when n1 = n2 -> raise (Failure (exceptf n1))
	| _ :: t -> helper t
	| [] -> ()
in helper (List.sort compare list)

let get_variable_list lst = 
	List.fold_left (fun lst (_,n) -> n::lst)
                        [] lst 

let get_function_list lst = 
	List.fold_left (fun lst fdecl -> fdecl.s_fname::lst)
                        [] lst
						
let rec exprl_to_sexprl (env,el) =
  let env_ref = ref(env) in
  let rec helper = function
	  head::tail ->
		let a_head, env = generate_sexpr(!env_ref,head) in
		env_ref := env;
		a_head::(helper tail)
	| [] -> []
  in (helper el), !env_ref


and check_call_type (env,fname,exprs) = 
	let actuals,_ = exprl_to_sexprl(env,exprs) in
	let fd = get_function_decl(env,fname) in 
	let match_params = fun (ft,_) actual -> 
		if (get_sexpr_type(actual) = ft || ft = Any ) then
			()
		else raise(Failure ("illegal actual argument found " ^ string_of_typ(get_sexpr_type(actual)) ^
			" expected " ^ string_of_typ ft ^ " in " ^ string_of_s_expr actual)) 
 		in		
	let _ = if List.length(exprs) != List.length(fd.formals) then
	raise (Failure ("expecting " ^ string_of_int
		(List.length fd.formals) ^ " argument(s) in " ^ string_of_expr(Call(fname,exprs))))
		else
			List.iter2 match_params fd.formals actuals
	 	in
			S_Call(fname,actuals,fd.typ)

and check_assign (env,e1,e2) =
	let sexpr1,_ = generate_sexpr(env,e1) in
	let sexpr2,_ = generate_sexpr(env,e2) in
	let type1 = get_sexpr_type (sexpr1) in
	let type2 = get_sexpr_type (sexpr2) in
		if (type1 = type2) then   (* 目前assign中array[]可以在左边但不能在右边 *)
			S_Assign(sexpr1,sexpr2,type1)
		else if (type1 = Float && type2 = Int) then
			S_Assign(sexpr1,S_Cast(Int,sexpr2,Float) ,type1)			
		else if (type1 = Arraytype(Int) && type2 = Int) then
			S_Assign(sexpr1,sexpr2,Int)
		else if (type1 = Arraytype(Bool) && type2 = Bool) then
			S_Assign(sexpr1,sexpr2,Bool)
		else if (type1 = Arraytype(Float) && type2 = Float) then
			S_Assign(sexpr1,sexpr2,Float)
		else if (type1 = Arraytype(String) && type2 = String) then
			S_Assign(sexpr1,sexpr2,String)
		else 
			raise(Failure ("illegal assignment " ^ string_of_typ type1 ^
								" = " ^ string_of_typ type2 ^ " in " ^ 
								string_of_s_expr sexpr1 ^ " = " ^ string_of_s_expr sexpr2))

and check_unop (env,op,e) =
	let sexpr,_ = generate_sexpr(env,e) in
	let type1 = get_sexpr_type(sexpr) in
	match(op) with
	Neg -> if (type1 = Int || type1 = Float) then S_Unop(op,sexpr,type1)
		else raise (Failure ("illegal unary operator " ^ string_of_uop op ^
				 string_of_typ(type1) ^ " in " ^ string_of_expr(Unop(op,e)) ))
		| Not -> if (type1 = Bool) then S_Unop(op,sexpr,type1)
		else raise (Failure ("illegal unary operator " ^ string_of_uop op ^
				 string_of_typ(type1) ^ " in " ^ string_of_expr(Unop(op,e)) ))
and check_binop (env,e1,op,e2) =
	let origin_sexpr1,_ = generate_sexpr(env,e1) in 
	let origin_sexpr2,_ = generate_sexpr(env,e2) in
	let origin_type1 = get_sexpr_type (origin_sexpr1) in
	let origin_type2 = get_sexpr_type (origin_sexpr2) in
	let (sexpr1,type1) = 
		if (origin_type1 = Int && origin_type2 = Float) then
			(S_Cast(Int,origin_sexpr1,Float),Float)
		else
			(origin_sexpr1,origin_type1) in
	let (sexpr2,type2) = 
		if (origin_type1 = Float && origin_type2 = Int) then
			(S_Cast(Int,origin_sexpr2,Float),Float)
		else
			(origin_sexpr2,origin_type2) in					
	match(op) with
	Add  when ((type1 = Int && type2 = Int) || (type1 = String && type2 = String) || (type1 = Float && type2 = Float))
	  -> S_Binop(sexpr1,op,sexpr2,type1)
		| Sub | Mult | Div when ((type1 = Int && type2 = Int) || (type1 = Float && type2 = Float))
		-> S_Binop(sexpr1,op,sexpr2,type1)
	    | Equal when (type1 = type2 && type1 = String) ->
		S_Binop(S_Literal(0),Equal,S_Call("strcmp",[sexpr1;sexpr2;],Int),Bool)
		| Neq when (type1 = type2 && type1 = String) ->
		S_Unop(Not,S_Binop(S_Literal(0) ,Equal,S_Call("strcmp",[sexpr1;sexpr2;],Int),Bool),Bool)
		| Equal | Neq when (type1 = type2) ->
			 S_Binop(sexpr1,op,sexpr2,Bool)
	    | Less | Leq | Greater | Geq when ((type1 = Int && type2 = Int) || (type1 = Float && type2 = Float) ) 
			-> S_Binop(sexpr1,op,sexpr2,Bool)
	    | And | Or when (type1 = Bool && type2 = Bool) -> S_Binop(sexpr1,op,sexpr2,Bool)
		| _ -> raise (Failure ("illegal binary operator " ^
				string_of_typ type1 ^ " " ^ string_of_op op ^ " " ^
				string_of_typ type2 ^ " in " ^ string_of_expr(Binop(e1,op,e2))))
and check_cast (env,t,e) =
	let sexpr,_ = generate_sexpr(env,e) in 
	let oldt = get_sexpr_type(sexpr) in
		match (oldt,t) with 
		 (Int,Int) | (Float,Float) | (Bool,Bool) | (String,String) 
		 | (Float,Int) | (Int,Float) | (String,Int) | (String,Float) 
		 | (Float,String) | (Int,String) | (Bool,String) ->
		S_Cast(oldt,sexpr,t)
		| _ ->
			raise(Failure("Covert from " ^ string_of_typ(oldt) ^ " to " ^ string_of_typ(t) ^ "is not permitted" ) )
and check_objaccess(env,e1,e2) =
	let sexpr1,_ = generate_sexpr(env,e1) in
	let objtyp = get_sexpr_type sexpr1 in
	match(objtyp) with 
		  	Object(cname) -> check_objaccess_class_property(env,cname,sexpr1,e1,e2)
		| 	_ -> raise (Failure ("illegal accessing:" ^ string_of_expr e1 ^ " is not an object"))
and check_objaccess_class_property(env,cname,sexpr1,e1,e2) =
	match(e2) with
			Id s -> check_objaccess_class_field(env,cname,sexpr1,s)
		|   Call(s, exprs) -> check_objaccess_class_method(env,cname,sexpr1,e1,e2,s,exprs)
		| 	ObjAccess(e3,e4) -> let e1' = ObjAccess(e1,e3) in check_objaccess(env,e1',e4)
		| 	_ -> raise (Failure ("illegal accessing:" ^ string_of_expr e2 ^ " is not a class property"))
and check_objaccess_class_field(env,cname,sexpr1,s) =
	let property_map = try StringMap.find cname env.env_class_map with | Not_found -> raise (Failure ("UndefinedClass" ^ cname)) in
	let e2_typ = try StringMap.find s property_map.field_map with | Not_found -> raise (Failure("illegal accessing: class " ^ cname ^ " has no field " ^ s)) in
	let sexpr2 = S_Id(s,e2_typ) in S_ObjAccess(sexpr1, sexpr2, e2_typ)
and check_objaccess_class_method(env,cname,sexpr1,e1,e2,s,exprs) =
	if cname = "constructor" then raise(Failure("illegal accessing: constructors cannot be accessed in this way")) else
	let property_map = try StringMap.find cname env.env_class_map with | Not_found -> raise (Failure ("UndefinedClass" ^ cname)) in
	let e2_fdecl = try StringMap.find s property_map.method_map with | Not_found -> raise (Failure("illegal accessing: class " ^ cname ^ " has no method " ^ s)) in
	let actuals,_ = exprl_to_sexprl(env,exprs) in
	let match_params = fun (ft,_) actual -> 
		if (get_sexpr_type(actual) = ft) then
			()
		else raise(Failure ("illegal actual argument found " ^ string_of_typ(get_sexpr_type(actual)) ^
			" expected " ^ string_of_typ ft ^ " in " ^ string_of_expr(Call(s,exprs))))
 		in	
	let _ = if List.length(exprs) != List.length(e2_fdecl.formals) then
		raise (Failure ("expecting " ^ string_of_int
		(List.length e2_fdecl.formals) ^ " argument(s) in " ^ string_of_expr e1 ^ "." ^ string_of_expr(Call(s,exprs))))
		else
			List.iter2 match_params e2_fdecl.formals actuals
	 	in
			S_ObjAccess(sexpr1,S_Call(s,actuals,e2_fdecl.typ),e2_fdecl.typ)

and is_typ_match(actuals,formals) = 
	(match actuals with [] -> Bool | h::l -> get_sexpr_type(h)) = (match formals with [] -> Bool | (ft,_)::l -> ft) &&
	is_typ_match((match actuals with [] -> [] | h::l -> l),(match formals with [] -> [] | h::l -> l))

and is_fdecl_match(env,fdecl,exprs) =
	if List.length(exprs) != List.length(fdecl.formals) then false else
	let actuals,_ = exprl_to_sexprl(env,exprs) in
	is_typ_match(actuals,fdecl.formals)

and find_and_test_constructor(fname,cname,sel) =
	let appended_constructor_name = cname ^ ".constructor" ^
		if List.length sel > 0 then
			"." ^ String.concat "." (List.map (fun sexpr -> string_of_typ (get_sexpr_type sexpr)) sel)
		else
			""
	in
	fname = appended_constructor_name

and check_objcreate(env,cname,el) =
	let property_map = try StringMap.find cname env.env_class_map with | Not_found -> raise (Failure ("UndefinedClass " ^ cname)) in

	let sel = List.rev(List.fold_left (fun sl e -> let sexpr, _ = generate_sexpr(env,e) in sexpr :: sl) [] el) in
	
	if StringMap.exists (fun k v -> find_and_test_constructor(k,cname,sel)) property_map.method_map then
		S_ObjCreate(cname,sel,Object(cname))
	else
		raise (Failure("No matching constructor for " ^ cname ^ "(" ^ String.concat "," (List.map string_of_expr el) ^ ")"))

and check_array_init(datatype,size) =
	S_ArrayCreate(datatype, size)

and check_array_access(env,e,index) =
	let se,_ = generate_sexpr(env, e) in
	let typ = match e with 
	  Id s -> get_id_type(env,s)
	in
	S_ArrayAccess(se, index, typ)

and check_delete(env,e) = 
	let se,_ = generate_sexpr(env,e) in 
	let t = get_sexpr_type(se) in
	match t with 
		Object(_) | Arraytype(_) -> S_Delete(se)
	| 	_ -> raise(Failure("illegal delete: " ^ string_of_expr e ^ " is not an object"))
and generate_sexpr (env,expr) =
	match (expr) with
		Literal i -> S_Literal(i), env
	| 	BoolLit b -> S_BoolLit(b), env
	| 	StringLit f -> S_StringLit(f), env
	| 	FloatLit f -> S_FloatLit(f), env
	| 	Id s -> S_Id(s, get_id_type(env,s)), env
	| 	Call(s, exprs) -> check_call_type(env,s,exprs), env
	| 	Assign(e1, e2) -> check_assign(env,e1,e2), env
	| 	Unop(op, e)-> check_unop(env,op,e), env
	| 	Binop(e1, op, e2)-> check_binop(env,e1,op,e2),env
	| 	Noexpr -> S_Noexpr,env
	| 	Nullexpr -> S_Null,env
	|	Cast(t,e) -> check_cast(env,t,e),env
    | 	ObjAccess(e1, e2) -> check_objaccess(env,e1,e2),env
	| 	ObjCreate(cname, el) -> check_objcreate(env,cname,el), env
	|   ArrayCreate(t, n) -> check_array_init(t,n),env
	|   ArrayAccess(e, index) -> check_array_access(env,e,index),env
	|   Delete(e) -> check_delete(env,e), env


let rec stmtl_to_sstmtl (env,stmt_list) = 
	let env_ref = ref(env) in
	let rec iter = function
	  	head::tail ->
			let a_head, env = generate_sstmt(!env_ref,head) in
			env_ref := env;
			a_head::(iter tail)
	| 	[] -> []
	in 
	let sstmt_list = (iter stmt_list), !env_ref in
	sstmt_list

and check_block (env,sl) =  match sl with
		[] -> S_Block([S_Expr(S_Noexpr,Void)]), env
	| 	_  -> 
		let s_sl, _ = stmtl_to_sstmtl(env,sl) in
		(S_Block(s_sl), env)	

and check_expr_stmt (env,e) =
	let se, env = generate_sexpr(env,e) in
	let t = get_sexpr_type(se) in 
	S_Expr(se, t), env
	
and check_return (env,e) = 
	let se, _ = generate_sexpr (env,e) in
	let t = get_sexpr_type(se) in
	if (t = env.env_return_type) then
		S_Return(se,t), env
	else 
		raise (Failure ("return gives " ^ string_of_typ t ^ " expected " ^
			string_of_typ env.env_return_type ^ " in " ^ string_of_s_expr se))

and check_if (env,p,b1,b2) = 
	let se, _ = generate_sexpr(env,p) in
	let t = get_sexpr_type(se) in
	let ifbody, _ = generate_sstmt (env,b1) in
	let elsebody, _ = generate_sstmt (env,b2) in
	if t = Bool
		then S_If(se, ifbody, elsebody), env
    else raise (Failure ("expected Boolean expression in " ^ string_of_s_expr se))
and check_for (env,e1,e2,e3,s) =
	let se1, _ = generate_sexpr(env,e1) in
	let se2, _ = generate_sexpr(env,e2) in
	let se3, _ = generate_sexpr(env,e3) in
	let forbody, _ = generate_sstmt(env,s) in
	let conditional = get_sexpr_type(se2) in
	let sfor = 
		if (conditional = Bool) then (* Could be void too *)
			S_For(se1, se2, se3, forbody)
		else raise (Failure ("invaild statement type in For"))			
	in
	(sfor, env)
and check_while (env,p,s) =
	let se, _ = generate_sexpr(env,p) in
	let t = get_sexpr_type(se) in
	let sstmt, _ = generate_sstmt(env,s) in 
	let swhile = 
		if (t = Bool ) then (* Could be void too *)
			S_While(se, sstmt)
		else raise (Failure ("invaild statement type in While"))			
	in
		(swhile, env)
and  generate_sstmt (env,stmt) = 
 match (stmt) with
		Block sl ->check_block(env,sl)
	| 	Expr e -> check_expr_stmt(env,e)
    |  	Return e -> check_return(env,e)           
    | 	If(p, b1, b2) -> check_if(env,p,b1,b2)
    | 	For(e1, e2, e3, st) -> check_for(env,e1,e2,e3,st)
    |   While(p, s) -> check_while(env,p,s)

let check_fbody(fname,fbody,return_type) =
	let len = List.length fbody in
	if len = 0 then () else 
	let final_stmt = List.hd (List.rev fbody) in 
	match return_type, final_stmt with
		Void, _ -> ()
	|	Constructortyp, _ -> ()
	| 	_, S_Return(_, _) -> ()
	| 	_ -> raise(Failure ("Missing return statement: " ^ fname))

(*
let check_foramls (fname, formals) = 
	if (List.length(List.sort_uniq(formals)) <> List.length(formals)) then
		raise(Failure ("duplicate formal in ")^fname) 
	else ()
*)

let generate_sfdecl (fdecl,function_map,global_map,class_map) =
	let local_map = List.fold_left (fun m (t,n) -> 
		if (t = Void) then
			raise(Failure("local " ^ n ^ " shouldn't be type void"))
		else
			StringMap.add n t m)
		StringMap.empty fdecl.locals in
	let param_map = List.fold_left (fun m (t,n) -> 
		if (t = Void) then
			raise(Failure("formal " ^ n ^ " shouldn't be type void"))
		else	
			StringMap.add n t m)
        StringMap.empty fdecl.formals in
	let env = 
	{
		env_class_map = class_map;
		env_global_map = global_map;
		env_local_map = local_map;
		env_param_map = param_map;
		env_function_map = function_map;
		env_return_type = fdecl.typ;
	} in
	let sfbody = List.fold_left (fun lst stmt -> 
		let sstmt, _ = generate_sstmt(env,stmt) in
		lst@[sstmt] ) [] fdecl.body in 
	ignore(check_fbody(fdecl.fname,sfbody,fdecl.typ));
	ignore(report_duplicate (fun n -> "duplicate formals " ^ n) (List.map snd fdecl.formals));
	ignore(report_duplicate (fun n -> "duplicate locals " ^ n) (List.map snd fdecl.locals));

	(*ignore(check_foramls(fdecl.fname,fdecl.formals));*)
	{
		s_typ = fdecl.typ;
		s_fname = fdecl.fname;
		s_formals = fdecl.formals;
		s_locals = fdecl.locals;
		s_body = sfbody;
	}

let get_main(functions) = 
	let find_main = (fun f -> match f.fname with "main" -> true  | _ -> false) in
	let mains = (List.find_all find_main functions) in
		if List.length mains < 1 then 
			raise (Failure("Main didn't defined"))
		else if List.length mains > 1 then 
			raise (Failure("MultipleMainsDefined")) 
		else List.hd mains

let append_to_name(cname,formals,name) =
	cname ^ "." ^ name ^ 
	if List.length formals > 0 then
		"." ^ String.concat "." (List.map (fun (typ,_) -> string_of_typ typ) formals)
	else

		""

let append_to_callname(formals,name) =
	name ^ 
	if List.length formals > 0 then
		"." ^ String.concat "." (List.map (fun (typ,_) -> string_of_typ typ) formals)
	else

		""

(* type s_expr =
    S_Literal of int
  | S_BoolLit of bool 
  | S_StringLit of string
  | S_FloatLit of float
  | S_Id of string * typ
  | S_Binop of s_expr * op * s_expr * typ
  | S_Unop of uop * s_expr * typ
  | S_Assign of s_expr * s_expr * typ
  | S_Call of string * s_expr list * typ
  | S_Noexpr
  | S_Null
  | S_ObjCreate of string * s_expr list * typ
  | S_ObjAccess of s_expr * s_expr * typ
  | S_Delete of s_expr *)

let rec implicit_this_in_constructor_sexpr(c_typ,s_expr,property_map) =
	match (s_expr) with 
		S_Id(s,typ) -> if StringMap.mem s property_map.field_map then S_ObjAccess(S_Id("this",c_typ),S_Id(s,typ),typ) else S_Id(s,typ)
	|   S_Binop(se1,op,se2,typ) -> S_Binop(implicit_this_in_constructor_sexpr(c_typ,se1,property_map),op,implicit_this_in_constructor_sexpr(c_typ,se2,property_map),typ)
	|   S_Unop(uop,se,typ) -> S_Unop(uop,implicit_this_in_constructor_sexpr(c_typ,se,property_map),typ)
	|  	S_Assign(se1,se2,typ) -> S_Assign(implicit_this_in_constructor_sexpr(c_typ,se1,property_map),implicit_this_in_constructor_sexpr(c_typ,se2,property_map),typ)
	|   S_Call(s,sel,typ) -> if StringMap.mem s property_map.method_map then 
								S_Call(s,List.rev (List.fold_left (fun lst se -> implicit_this_in_constructor_sexpr(c_typ,se,property_map) :: lst) [] sel),typ) 
							 else S_Call(s,sel,typ)
	|   S_ObjCreate(s,sel,typ) -> S_ObjCreate(s,List.rev (List.fold_left (fun lst se -> implicit_this_in_constructor_sexpr(c_typ,se,property_map) :: lst) [] sel),typ)
	|   S_ObjAccess(se1,se2,typ) -> (match se1 with
									S_Id("this",_) -> S_ObjAccess(se1,se2,typ)
								| 	_ -> S_ObjAccess(implicit_this_in_constructor_sexpr(c_typ,se1,property_map),implicit_this_in_constructor_sexpr(c_typ,se2,property_map),typ))
	|  	S_Delete(se) -> S_Delete(implicit_this_in_constructor_sexpr(c_typ,se,property_map))
	|   _ -> s_expr


let rec implicit_this_in_constructor_stmt(c_typ,s_stmt,property_map) = 
	match (s_stmt) with
		S_Block(sl) -> S_Block(implicit_this_in_constructor_body(c_typ,sl,property_map))
	|  	S_Expr(s,t) -> S_Expr(implicit_this_in_constructor_sexpr(c_typ,s,property_map),t)
	|   S_Return(_,_) -> raise (Failure("Constructors cannot return a value"))
	|  	S_If(s,st1,st2) -> S_If(implicit_this_in_constructor_sexpr(c_typ,s,property_map),implicit_this_in_constructor_stmt(c_typ,st1,property_map),
								implicit_this_in_constructor_stmt(c_typ,st2,property_map))
	|  	S_For(s1,s2,s3,st) -> S_For(implicit_this_in_constructor_sexpr(c_typ,s1,property_map),implicit_this_in_constructor_sexpr(c_typ,s2,property_map),
									implicit_this_in_constructor_sexpr(c_typ,s3,property_map),implicit_this_in_constructor_stmt(c_typ,st,property_map))
	|  	S_While(s,st) -> S_While(implicit_this_in_constructor_sexpr(c_typ,s,property_map),implicit_this_in_constructor_stmt(c_typ,st,property_map))

and implicit_this_in_constructor_body (c_typ,s_body,property_map) = 
	let modified_s_body = 
		List.fold_left (fun lst s_stmt -> implicit_this_in_constructor_stmt(c_typ,s_stmt,property_map)::lst) [] s_body 
	in
	List.rev modified_s_body

let implicit_constructor_body (c_typ,s_body,property_map) =
	[S_Expr(S_Assign(
		S_Id("this",c_typ),
		S_Call("cast",
				[
					S_Call("malloc",
							[
								S_Call("sizeof",
										[
											S_Id("this",c_typ)
										],Int)
							],Any)
				],c_typ),
		c_typ
	),c_typ)]
	@ implicit_this_in_constructor_body(c_typ,s_body,property_map) @ 
	[S_Return(S_Id("this",c_typ),c_typ)]


let update_sconstrdecl(sfdecl,cname,property_map) =
	{
		s_typ = Object(cname);
		s_fname = sfdecl.s_fname;
		s_formals = sfdecl.s_formals;
		s_locals =  sfdecl.s_locals;
		s_body = implicit_constructor_body(Object(cname),sfdecl.s_body,property_map);
	}

let update_sfdecl(sfdecl,cname) =
	{
		s_typ = sfdecl.s_typ;
		s_fname = cname ^ "." ^ sfdecl.s_fname;
		s_formals = sfdecl.s_formals;
		s_locals = sfdecl.s_locals;
		s_body = sfdecl.s_body;
	}

let generate_scdecl (cdecl,class_map) =
	let property_map = try StringMap.find cdecl.cname class_map with | Not_found -> raise (Failure ("UndefinedClass " ^ cdecl.cname)) in
	{
		s_cname = cdecl.cname;
		s_fields = cdecl.cbody.fields;
		s_constructors = (let raw_s_constructors = List.fold_left (fun lst fdecl -> 
							generate_sfdecl(fdecl,property_map.method_map,property_map.field_map,class_map)::lst) 
							[] cdecl.cbody.constructors in 
							List.fold_left (fun lst sfdecl -> 
						 	update_sconstrdecl(sfdecl,cdecl.cname,property_map)::lst)
						 	[] raw_s_constructors);
		s_methods = (let raw_s_methods = List.fold_left (fun lst fdecl -> 
							generate_sfdecl(fdecl,property_map.method_map,property_map.field_map,class_map)::lst)
							[] cdecl.cbody.methods in
							List.fold_left (fun lst sfdecl ->
							update_sfdecl(sfdecl,cdecl.cname)::lst)
							[] raw_s_methods);
	}

let generate_sast (globals,functions,builtins,classes,function_map,global_map,class_map) =
	let smain = generate_sfdecl(get_main(functions),function_map,global_map,class_map) in 
	let sfdecls = List.fold_left (fun lst fdecl -> 
		if (fdecl.fname = "main") then lst else
		generate_sfdecl(fdecl,function_map,global_map,class_map)::lst) [] functions in
	let sbuiltins = List.fold_left (fun lst fdecl -> generate_sfdecl(fdecl,function_map,global_map,class_map)::lst) [] builtins in
	let scdecls = List.fold_left (fun lst cdecl -> generate_scdecl(cdecl,class_map)::lst) [] classes in
	(globals,sfdecls,scdecls,smain,sbuiltins)

let overload_class(classes) =
	List.fold_left (fun lst cdecl ->
	{
		cname = cdecl.cname;
		cbody = 
		{
			fields = cdecl.cbody.fields;
			methods = List.fold_left (fun lst fdecl -> {
				typ = fdecl.typ;

				fname = append_to_callname((Object(cdecl.cname),"this")::fdecl.formals,fdecl.fname);

				formals = (Object(cdecl.cname),"this")::fdecl.formals;
				locals = fdecl.locals;
				body = fdecl.body;
			}::lst) [] cdecl.cbody.methods;
			constructors = List.fold_left (fun lst condecl -> {
					typ = condecl.typ;
    				fname = append_to_name(cdecl.cname,condecl.formals,condecl.fname);
    				formals = condecl.formals;
    				locals = (Object(cdecl.cname),"this") :: condecl.locals;
   					body = condecl.body;
			}::lst) [] cdecl.cbody.constructors;
		}
	}::lst)
	[] classes

let overload_callname(s,el,env) =
	let newS = 
	s ^
	if List.length el > 0 then
		"." ^ String.concat "." (List.map (fun expr -> let sexpr,_ = generate_sexpr(env,expr) in string_of_typ(get_sexpr_type(sexpr))) el)
	else 
		""
	in 
	newS

let rec overload_expr(expr,env) =
	match (expr) with 
		ObjAccess(e1,e2) -> (
			match e2 with
				Call(s,el) -> let modified_el = List.rev(List.fold_left(fun lst expr -> overload_expr(expr,env)::lst) [] (e1::el)) in
							  ObjAccess(e1,Call(overload_callname(s,modified_el,env),modified_el))
			|   ObjAccess(e3,e4) -> let e1' = ObjAccess(e1,e3) in overload_expr(ObjAccess(e1',e4),env)
			| 	_ -> expr
		)
	| 	Binop(e1,op,e2) -> Binop(overload_expr(e1,env),op,overload_expr(e2,env))
	|   Unop(uop,e) -> Unop(uop,overload_expr(e,env))
	|  	Assign(e1,e2) -> Assign(overload_expr(e1,env),overload_expr(e2,env))
	| 	Call(s,el) -> Call(s,List.rev (List.fold_left (fun lst e -> overload_expr(e,env)::lst)[] el))
	| 	ObjCreate(s,el) -> ObjCreate(s,List.rev (List.fold_left (fun lst e -> overload_expr(e,env)::lst)[] el))
	| 	_ -> expr

let rec overload_stmt(stmt,env) =
	match (stmt) with
		Block(sl) -> Block(overload_function_body(sl,env))
	| 	Expr(e) -> Expr(overload_expr(e,env))
	| 	Return(e) -> Return(overload_expr(e,env))
	|   If(e,st1,st2) -> If(overload_expr(e,env),overload_stmt(st1,env),overload_stmt(st2,env))
	|   For(e1,e2,e3,st) -> For(overload_expr(e1,env),overload_expr(e2,env),overload_expr(e3,env),overload_stmt(st,env))
	| 	While(e,st) -> While(overload_expr(e,env),overload_stmt(st,env))

and overload_function_body(body,env) =
	let modified_body = 
		List.fold_left (fun lst stmt -> overload_stmt(stmt,env)::lst) [] body
	in
	List.rev modified_body

let overload_function(functions,function_map,global_map,class_map) =
	List.fold_left (fun lst fdecl ->
	let local_map = List.fold_left (fun m (t,n) -> 
		if (t = Void) then
			raise(Failure("local " ^ n ^ " shouldn't be type void"))
		else
			StringMap.add n t m)
		StringMap.empty fdecl.locals in
	let param_map = List.fold_left (fun m (t,n) -> 
		if (t = Void) then
			raise(Failure("formal " ^ n ^ " shouldn't be type void"))
		else	
			StringMap.add n t m)
        StringMap.empty fdecl.formals in
	let env = 
	{
		env_class_map = class_map;
		env_global_map = global_map;
		env_local_map = local_map;
		env_param_map = param_map;
		env_function_map = function_map;
		env_return_type = fdecl.typ;
	} in 
	{
		typ = fdecl.typ;
		fname = fdecl.fname;
		formals = fdecl.formals;
		locals = fdecl.locals;
		body = overload_function_body(fdecl.body,env);
	}::lst)
	[] functions


let overload_body_class(classes,class_map) =
	List.fold_left (fun lst cdecl ->
	let property_map = try StringMap.find cdecl.cname class_map with | Not_found -> raise  (Failure ("UndefinedClass " ^ cdecl.cname)) in
	{
		cname = cdecl.cname;
		cbody = 
		{
			fields = cdecl.cbody.fields;
			methods = overload_function(cdecl.cbody.methods,property_map.method_map,property_map.field_map,class_map);
			constructors = overload_function(cdecl.cbody.constructors,property_map.method_map,property_map.field_map,class_map);
		}
	}::lst)
	[] classes

(* Main method for analyzer *)
let analyze program = 
match program with
	(globals, functions, classes) -> 
		let builtins = define_builtin_functions in
		let modified_classes = overload_class(classes) in
		let function_map, global_map = build_maps(functions,builtins,globals) and class_map = build_class_map(globals,functions@builtins,modified_classes) in

		let modified_body_classes = overload_body_class(modified_classes,class_map) in 
		let modified_functions = overload_function(functions,function_map,global_map,class_map) in
		let modified_function_map, _ = build_maps(modified_functions,builtins,globals) in
			generate_sast(globals,modified_functions,builtins,modified_body_classes,modified_function_map,global_map,class_map)

