open Ast
type s_expr =
    S_Literal of int
  | S_BoolLit of bool 
  | S_StringLit of string
  | S_FloatLit of float
  | S_Id of string * typ
  | S_Binop of s_expr * op * s_expr * typ
  | S_Unop of uop * s_expr * typ
  | S_Assign of s_expr * s_expr * typ
  | S_Call of string * s_expr list * typ
  | S_Noexpr
  | S_Null
  | S_Cast of typ * s_expr * typ
  | S_ObjCreate of string * s_expr list * typ
  | S_ObjAccess of s_expr * s_expr * typ
  | S_ArrayCreate of typ * int
  | S_ArrayAccess of s_expr * int * typ
  | S_Delete of s_expr

type s_stmt =
    S_Block of s_stmt list
  | S_Expr of s_expr * typ
  | S_Return of s_expr * typ
  | S_If of s_expr * s_stmt * s_stmt
  | S_For of s_expr * s_expr * s_expr * s_stmt
  | S_While of s_expr * s_stmt

type s_func_decl = {
    s_typ : typ;
    s_fname : string;
    s_formals : bind list;
    s_locals : bind list;
    s_body : s_stmt list;
}

type s_class_decl = {
    s_cname: string;
    s_fields: bind list;
    s_constructors: s_func_decl list;
    s_methods: s_func_decl list;
}

type s_program = {
    global_vars: bind list;
    functions: s_func_decl list;
    classes: s_class_decl list;
    main: s_func_decl;
    builtins: s_func_decl list;
}

(* Pretty-printing functions *)
let rec string_of_ast_typ = function
    Int -> "int" 
    | Bool -> "bool"
    | Void -> "void"
    | Null -> "null"
    | String -> "string"
    | Object(name) -> name
    | Arraytype(t) -> "array of " ^ string_of_ast_typ t 
    | Constructortyp -> "constructor"
    | Any -> "any"

let rec string_of_expr_type = function
    S_Literal(_) -> "int"
  | S_BoolLit(_) -> "bool" 
  | S_StringLit(_) -> "string"
  | S_Id(name,data_type) -> string_of_ast_typ data_type
  | S_Binop(lhr,op,rhs,data_type) ->  string_of_ast_typ data_type
  | S_Unop(op,exp,data_type) -> string_of_ast_typ data_type
  | S_Assign(lhs,rhs,data_type) -> string_of_ast_typ data_type
  | S_Call(fname,expr_list,data_type) -> string_of_ast_typ data_type
  | S_Noexpr -> "Noexpr"
  | S_Null -> "null"
  | S_ObjCreate(obj_name,expr_list,data_type) -> string_of_ast_typ data_type
  | S_ObjAccess(obj_name,field,data_type) -> string_of_ast_typ data_type
  | S_ArrayCreate(typ, len) -> "(array of type " ^ string_of_ast_typ typ ^ " with length " ^ string_of_int len ^ ")"
  | S_ArrayAccess(arrayName, index, data_type) ->  "yuanlaishitype..type: " ^ string_of_ast_typ data_type ^ " index: " ^ string_of_int index ^ ")"
  | S_Delete(expr) -> string_of_expr_type expr
  | S_Cast(oldt,expr,newt) -> string_of_ast_typ newt

let rec string_of_s_expr = function
    S_Literal(l) -> string_of_int l
  | S_BoolLit(true) -> "true"
  | S_BoolLit(false) -> "false" 
  | S_StringLit(s) -> s
  | S_FloatLit(f) -> string_of_float f
  | S_Id(s,t) -> "$" ^ s ^ "[" ^ string_of_typ t ^ "]" 
  | S_Binop(e1, o, e2, t) ->
      "{" ^ string_of_s_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_s_expr e2 ^ "}[" ^ string_of_typ t ^ "]" 
  | S_Unop(o, e,t) -> string_of_uop o ^ string_of_s_expr e ^ "[" ^ string_of_typ t ^ "]" 
  | S_Assign(e1,e2,t) -> string_of_s_expr e1 ^ " = " ^ string_of_s_expr e2 ^ "[" ^ string_of_typ t ^ "]" 
  | S_Call(f, el,t) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_s_expr el) ^ ")" ^ "[" ^ string_of_typ t ^ "]" 
  | S_Noexpr -> ""
  | S_Null -> "Null"
  | S_Cast(oldt,expr,newt) ->
     "{" ^ "(" ^ string_of_s_expr(expr) ^ ")" ^ "[" ^ string_of_typ(oldt) ^ "->" ^ string_of_typ(newt) ^ "]" ^ "}"
  | S_ObjAccess(s1,s2,t) -> "{Object Access: " ^ string_of_s_expr s1 ^ "." ^ string_of_s_expr s2 ^ " " ^ string_of_typ t ^ "}"
  | S_ObjCreate(s,sel,t) -> "{Object Create: new " ^ s ^ "(" ^ String.concat "," (List.map string_of_s_expr sel) ^ ")[" ^ string_of_typ t ^ "]}"
  | S_ArrayCreate(typ, len) -> "(array of type " ^ string_of_typ typ ^ " with length " ^ string_of_int len ^ ")"
  | S_ArrayAccess(arrayName, index, t) -> 
        (* match arrayExpr with S_Id(arr,t) -> "(array_name: " ^ arr ^ "type:" ^ string_of_typ t ^ " index: " ^ string_of_int index ^ ")" *)
        "(array_name: " ^ string_of_s_expr arrayName ^ " type: " ^ string_of_typ t ^ " index: " ^ string_of_int index ^ ")" 
  | S_Delete(se) -> "delete " ^ string_of_s_expr se

let rec string_of_s_stmt = function
    S_Block(s_stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_s_stmt s_stmts) ^ "}\n"
  | S_Expr(s_expr,t) -> "{" ^ string_of_s_expr s_expr ^ "} [" ^ string_of_typ t ^ "]" ^ ";\n" ;
  | S_Return(s_expr,t) -> "return " ^ string_of_s_expr s_expr ^ ";\n" ^ "[" ^ string_of_typ t ^ "]";
  | S_If(e, s, S_Block([])) -> "if (" ^ string_of_s_expr e ^ ")\n" ^ string_of_s_stmt s
  | S_If(e, s1, s2) ->  "if (" ^ string_of_s_expr e ^ ")\n" ^
      string_of_s_stmt s1 ^ "else\n" ^ string_of_s_stmt s2
  | S_For(e1, e2, e3, s) ->
      "for (" ^ string_of_s_expr e1  ^ " ; " ^ string_of_s_expr e2 ^ " ; " ^
      string_of_s_expr e3  ^ ") " ^ string_of_s_stmt s
  | S_While(e, s) -> "while (" ^ string_of_s_expr e ^ ") " ^ string_of_s_stmt s

let string_of_s_vdecl (t, id) = string_of_typ t ^ " " ^ id ^ ";\n"

let string_of_s_formal (t, id) = string_of_typ t ^ " " ^ id

let string_of_s_fdecl fdecl =
  string_of_typ fdecl.s_typ ^ " " ^
  fdecl.s_fname ^ "(" ^ String.concat ", " (List.map string_of_s_formal fdecl.s_formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.s_locals)^ 
  String.concat "" (List.map string_of_s_stmt fdecl.s_body) ^
  "\n}\n"

let string_of_s_cdecl cdecl = 
    "class" ^ " " ^ cdecl.s_cname ^ " { \n" ^ 
    String.concat "" (List.map string_of_s_vdecl cdecl.s_fields) ^
    String.concat "" (List.map string_of_s_fdecl cdecl.s_constructors) ^
    String.concat "" (List.map string_of_s_fdecl cdecl.s_methods) ^
    "}\n"

let string_of_s_program (vars, funcs, classes, main, builtins) =
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_s_fdecl funcs) ^ "\n" ^
  String.concat "\n" (List.map string_of_s_cdecl classes) ^ "\n" ^
  String.concat "\n" (List.map string_of_s_fdecl builtins) ^ "\n" ^
  (string_of_s_fdecl main) ^ "\n"
