
open Llvm

module L = Llvm
module A = Ast
module S = Sast


module StringMap = Map.Make(String)

module H = Hashtbl

module P=Printf
let counter : int ref = ref 0
let object_types:(string, L.lltype) H.t = H.create 10;;
let object_field_indices:(string, int) H.t = H.create 50;;
let globals_table:(string, L.llvalue) H.t = H.create 50;;
let locals_table:(string, L.llvalue) H.t = H.create 50;;
let params_table:(string, L.llvalue) H.t = H.create 50;;
let new_obj_table:(int, L.llvalue) H.t = (H.create 50);;
let func_table:(string, L.llvalue * S.s_func_decl) H.t = H.create 50;;

let log_to_file str=
ignore()

let print_hashtable_int table = 
ignore()

let print_hashtable table = 
  H.iter (fun k v -> log_to_file (k^(L.string_of_llvalue v)^"\n")) table;;

let context = L.global_context ()
let the_module = L.create_module context "CGC"
and i32_t  = L.i32_type  context (* integer *)
and i8_t   = L.i8_type   context 
and i1_t   = L.i1_type   context (* boolean *)
and i8_pt  = L.pointer_type (L.i8_type context) (* string *)
and void_t = L.void_type context  (* void *)
and f_t = L.double_type context;;


let rec find_object_type name = 
  try Hashtbl.find object_types name
  with | Not_found -> raise (Failure("no class def of " ^ name))

(* convert sast type to underlying *)
and get_underlying_type_of_sexpr = function
    S.S_Literal(_) -> i32_t
  | S.S_BoolLit(_)  -> i1_t
  | S.S_FloatLit(_) -> f_t
  | S.S_StringLit(_) -> i8_pt
  | S.S_Id(_, data_type) -> ltype_of_typ data_type
  | S.S_Unop(_, _, data_type) -> ltype_of_typ data_type
  | S.S_Binop(_, _, _, data_type) -> ltype_of_typ data_type
  | S.S_Assign(_, _, data_type) -> ltype_of_typ data_type
  | S.S_Call(_, _, data_type) -> ltype_of_typ data_type
  | S.S_Noexpr -> void_t
  | S.S_Cast(_,_,data_type) ->  ltype_of_typ data_type
  | S.S_ObjAccess(_, _, data_type)		-> ltype_of_typ data_type
  | S.S_ObjCreate(_, _, data_type) 	->  ltype_of_typ data_type
  (*| S.SArrayPrimitive(_, d)	-> d*)
  | S.S_Null -> i32_t
  | 	d -> raise(Failure ("Met unknown type in get_underlying_type_of_sexpr\n") ) 

and get_ast_type_of_sexpr = function
    S.S_Literal(_) -> A.Int
  | S.S_BoolLit(_)  -> A.Bool
  | S.S_FloatLit(_) -> A.Float
  | S.S_StringLit(_) -> A.String
  | S.S_Id(_, data_type) -> data_type
  | S.S_Unop(_, _, data_type) -> data_type
  | S.S_Binop(_, _, _, data_type) -> data_type
  | S.S_Assign(_, _, data_type) -> data_type
  | S.S_Call(_, _, data_type) -> data_type
  | S.S_Noexpr -> A.Void
  | S.S_ObjAccess(_, _, data_type)		-> data_type
  | S.S_ObjCreate(_, _, data_type) 	->  data_type
  | S.S_Cast(_,_,data_type) ->  data_type
  (*| S.SArrayPrimitive(_, d)	-> d*)
  | S.S_ArrayAccess(arrayName, index, t) -> t
  | S.S_ArrayCreate(typ, size) -> typ
  | S.S_Null -> A.Int
  | 	d -> raise(Failure ("Met unknown type in get_ast_type_of_sexpr\n") ) 

and ltype_of_typ = function
    A.Int -> i32_t
  | A.Bool -> i1_t
  | A.Void -> void_t 
  | A.String -> i8_pt
  | A.Float -> f_t
  | Object(name) ->  L.pointer_type(find_object_type name)
  | A.Arraytype(t) -> L.pointer_type (ltype_of_typ t)
  | 	d -> raise(Failure ("Met unknown type in ltype_of_typ\n") ) 

(* Return the value for a variable or formal or global argument *)
and lookup the_name = 
  try H.find locals_table the_name
  with | Not_found ->
    (try  H.find params_table the_name
     with | Not_found -> 
       (try H.find globals_table the_name
        with | Not_found -> raise (Failure("Unbound var " ^ the_name))
       )
    )

(* Get the built function from the module *)
and  func_lookup fname = 
  match (L.lookup_function fname the_module) with
    None 	-> raise (Failure ("No function "^fname) )
  | Some f 	-> f

let rec codegen_globals globals =
  ignore(log_to_file "Codegen globals\n");

  (* Declare each global variable; remember its value in a map *)
  let add_to_global_table table (t, n) =
    let init = L.const_int (ltype_of_typ t) 0 in 
    H.add table n (L.define_global n init the_module)
  in  
  ignore(List.map (add_to_global_table globals_table) globals);
  ignore(print_hashtable globals_table);
  log_to_file "Generated globals\n"

(* Define each function (arguments and return type) so we can call it *)
and codegen_function_decls functions = 
  ignore(log_to_file "Codegen function declarations\n");
  ignore(List.map (fun f -> log_to_file ("This time build function "^f.S.s_fname^"\n") ) functions);

  let add_to_func_table table fdecl =
    let name = fdecl.S.s_fname
    and formal_types =
      Array.of_list (List.map (fun (t,_) -> ltype_of_typ t) fdecl.S.s_formals)
    in 
    let ftype = L.function_type (ltype_of_typ fdecl.S.s_typ) formal_types in
    H.add table name (L.define_function name ftype the_module, fdecl) 
  in
  ignore(List.map (add_to_func_table func_table) functions);
  log_to_file "Generated function declarations\n"


and codegen_builtins func_decls = 
  ignore(log_to_file "Codegen builtins\n");
  (* Declare printf(), which the print built-in function will call *)
  let printf_t = L.var_arg_function_type i32_t [| L.pointer_type i8_t |] in
  let printf_func = L.declare_function "printf" printf_t the_module in

  (* Declare malloc *)
  let malloc_ty = L.function_type (i8_pt) [| i32_t |] in
  let malloc_func = L.declare_function "malloc" malloc_ty the_module in

    (* Declare free *)
  let free_ty = L.function_type (void_t) [| i8_pt |] in   
  let free_func = L.declare_function "free" free_ty the_module in

  (* Dump the module to check builtin functions *)
  ignore(log_to_file "Generated builtin functions\n")
(* L.dump_module the_module *)

(* Generate classes *)
and  codegen_struct_stub class_struct =
  let struct_t = L.named_struct_type context class_struct.S.s_cname in
  H.add object_types class_struct.S.s_cname struct_t


and codegen_struct class_struct = 
  let struct_t = H.find object_types class_struct.S.s_cname in
  let type_list = List.map (fun (data_type,name) -> ltype_of_typ data_type) class_struct.S.s_fields in
  let name_list = List.map (fun (data_type, name) -> name) class_struct.S.s_fields in

  let type_array = (Array.of_list type_list) in
  List.iteri (fun index field ->
      let n = class_struct.S.s_cname ^ "." ^ field in
      H.add object_field_indices n index;
    ) name_list;
  L.struct_set_body struct_t type_array true

(* Pass constructor to this function *)
and codegen_obj_create fname expr_list data_type llbuilder = 
  (* Call the full name of the function *)
  let class_name = match data_type with
      A.Object(obj_name) -> obj_name
    | d -> raise (Failure ("Data type in Obj_Create is not an object but "^(A.string_of_typ d) ) )
  in
  let string_of_params params = 
    String.concat "." (List.map (fun p -> (S.string_of_expr_type p ) ) params )
  in
  let param_str = (match expr_list with
        [] -> ""
      | _ -> "." ^ (string_of_params expr_list)
    )in
  let full_name = (class_name ^ ".constructor"^param_str) in
  let f = func_lookup full_name in
  let generate_param_func builder isReturn param =
    match param with 
      S.S_Id(id,A.Object(obj_name)) -> 
      lookup id
    | _ -> 
      codegen_sexpr llbuilder false param
  in
  let params = List.map (generate_param_func llbuilder false) expr_list in
  let obj = L.build_call f (Array.of_list params) "tmp" llbuilder in
  obj

and codegen_id id d llbuilder =
  try  (
    let _val = H.find locals_table id in
    ignore(log_to_file ("Found "^id^" in local table\n") );
    L.build_load _val id llbuilder
  )
  with | Not_found ->
    (try let _val = H.find params_table id in
       ignore(log_to_file ("Found "^id^" in params\n") );
       _val
     with | Not_found -> 
       (try let _val = H.find globals_table id in
          ignore(log_to_file ("Found "^id^" in global table\n") );
          L.build_load _val id llbuilder
        with | Not_found -> raise (Failure("Unbound var " ^ id))
       )
    ) 
and codegen_id_obj_in_param id d llbuilder = 
  lookup id

(* Very possibly we need to rewrite this *)
and codegen_obj_access isAssign the_obj the_field data_type llbuilder = 
  ignore(log_to_file "Building obj access\n");
  ignore(log_to_file ((S.string_of_s_expr the_obj)^"\n") );
  ignore(log_to_file ((S.string_of_s_expr the_field)^"\n") );

  ignore(log_to_file "Now we need object field index table\n");
  ignore(print_hashtable_int object_field_indices);

  (* LHS can be ID or a nested Obj_Access *)
  ignore(log_to_file "Generating parent\n");
  let codegen_lhs = function
      S.S_Id(id, data_type) -> 
      ignore(log_to_file "LHS of obj access is just the obj\n");
      let _found = lookup id in
      _found
    | S.S_ArrayAccess(arrayName, index, t)	-> generate_array_access true arrayName index llbuilder 
    | S.S_ObjAccess(o,f,t)-> 
      (* Need to load lhs before accessing further *)
      let _val = codegen_obj_access isAssign o f t llbuilder in
      L.build_load _val "load_nested_obj" llbuilder
    | 	se 	-> 
      ignore(log_to_file "Got unhandled LHS in obj access:\n");
      ignore(log_to_file ("Type is "^(S.string_of_s_expr se)^"\n" ) );
      raise( Failure "Unhandled case in LHS in obj_access\n ")
  in
  let parent = codegen_lhs the_obj in
  ignore(log_to_file ("Generated parent: "^(L.string_of_llvalue parent)^"\n") );

  (* RHS can be ID *)
  ignore(log_to_file "Generating field\n");

  let rec codegen_rhs (parent_expr:L.llvalue) (parent_type:A.typ) field_expr= 
    match field_expr with
      S.S_Id(id,data_type) -> 
      (
        (* Find type of parent *)
        let search_term = ((S.string_of_ast_typ parent_type) ^ "." ^ id) in
        let field_index = Hashtbl.find object_field_indices search_term in
        let _val = L.build_struct_gep parent_expr field_index id llbuilder in
        let _mat = match data_type with 
            A.Object(name)->
            let _store = 
              if isAssign then
                _val
              else begin 
                let _load = L.build_load _val id llbuilder in
                _load
              end
            in
            _store
          | _ -> 
              _val
        in
        _mat
      )
    | S_ArrayAccess(arrayName, index, t) ->
      let ce  = codegen_rhs parent_expr parent_type arrayName in
      let index = L.const_int i32_t index in
      let index = L.build_add index (L.const_int i32_t 1) "temp_afterAdd1" llbuilder in
      let _val = build_gep ce [| index |] "tmp" llbuilder in
      if isAssign
          then _val
          else build_load _val "tmp" llbuilder 

    | S.S_Call (fname, act, _) ->
      ignore(log_to_file ("Build object function "^fname^" with expr: \n") );
      ignore(List.map (fun expr -> log_to_file ((S.string_of_s_expr expr) ^ "\n") ) act );
      let class_name = A.string_of_typ parent_type in
      let func_name = class_name^"."^fname in
      ignore(log_to_file ("Function name is "^func_name^"\n"));
      let (fdef, fdecl) = H.find func_table func_name in
      let actuals = List.rev (List.map (codegen_sexpr llbuilder true) (List.rev act)) in
      let result = (match fdecl.S.s_typ with A.Void -> ""
                                           | _ -> func_name ^ "_result") in
      L.build_call fdef (Array.of_list actuals) result llbuilder
    | se ->
      ignore(log_to_file "Got unhandled RHS in obj access:\n");
      ignore(log_to_file ("Type is "^(S.string_of_s_expr se)^"\n" ) );
      raise( Failure "Unhandled case in RHS in obj_access\n ")
  in


  let lhs_type = get_ast_type_of_sexpr the_obj in 
  let lhs = codegen_lhs the_obj in
  let rhs = codegen_rhs lhs lhs_type the_field in
  rhs

(* Check type of obj field *)

and get_value deref vname builder = 
  if deref then
    let var = try H.find locals_table vname with 
      | Not_found -> try H.find params_table vname with 
        | Not_found -> try H.find globals_table vname with
          | Not_found -> raise (Failure("unknown variable name " ^ vname))
    in
    L.build_load var vname builder

  else
    let var = try H.find locals_table vname with 
      | Not_found -> try H.find params_table vname with 
        | Not_found -> try H.find globals_table vname with
          | Not_found -> raise (Failure("unknown variable name " ^ vname))
    in
    var

and initialise_array arr arr_len init_val start_pos llbuilder =
  let new_block label =
    let f = L.block_parent (L.insertion_block llbuilder) in
    L.append_block (context) label f
  in
  let bbcurr = L.insertion_block llbuilder in
  let bbcond = new_block "array.cond" in
  let bbbody = new_block "array.init" in
  let bbdone = new_block "array.done" in
  ignore (L.build_br bbcond llbuilder);
  L.position_at_end bbcond llbuilder;

  (* Counter into the length of the array *)
  let counter = L.build_phi [L.const_int i32_t start_pos, bbcurr] "counter" llbuilder in
  add_incoming ((L.build_add counter (L.const_int i32_t 1) "tmp" llbuilder), bbbody) counter;
  let cmp = build_icmp Icmp.Slt counter arr_len "tmp" llbuilder in
  ignore (L.build_cond_br cmp bbbody bbdone llbuilder);
  position_at_end bbbody llbuilder;

  (* Assign array position to init_val *)
  let arr_ptr = build_gep arr [| counter |] "tmp" llbuilder in
  ignore (build_store init_val arr_ptr llbuilder);
  ignore (build_br bbcond llbuilder);
  position_at_end bbdone llbuilder

(* Generate 1 dimension array *)
and generate_one_d_array typ size builder =
  let t = ltype_of_typ typ in

  let size = (L.const_int i32_t size) in

  let size_t = L.build_intcast (L.size_of t) i32_t "1tmp" builder in

  let size = L.build_mul size_t size "2tmp" builder in  (* size * length *)
  let size_real = L.build_add size (L.const_int i32_t 1) "arr_size" builder in

  let arr = L.build_array_malloc t size_real "333tmp" builder in
  let arr = L.build_pointercast arr (L.pointer_type t) "4tmp" builder in

  let arr_len_ptr = L.build_pointercast arr (L.pointer_type i32_t) "5tmp" builder in

  ignore(L.build_store size_real arr_len_ptr builder); 
  initialise_array arr_len_ptr size_real (L.const_int i32_t 0) 0 builder;
  arr 

and generate_array_access deref arrayName index builder = 
  (* let _ = print_int index in *)
  let index = L.const_int i32_t index in
  let index = L.build_add index (L.const_int i32_t 1) "temp_afterAdd1" builder in
  let arr = match arrayName with
    | S.S_Id(name, _) -> get_value true name builder
    | _ -> raise(Failure("No such arrayName:"))
  in
  let _val = L.build_gep arr [| index |] "2tmp" builder in
  if deref
  then L.build_load _val "3tmp" builder 
  else _val 

(* Construct code for an expression; return its value *)
and codegen_assign lhs rhs llbuilder =
  ignore(log_to_file ("Building assignment: "^(S.string_of_s_expr lhs)^" = "^(S.string_of_s_expr rhs)^"\n" ) );
  (match lhs with
     S.S_Id (s,t) ->  
     (* ignore(print_string ("print type is " ^ A.string_of_typ t)); *)
     let e2' = codegen_sexpr llbuilder false rhs in 
     if (String.compare s "this")=0 then begin
       ignore(log_to_file ("LHS of assign is "^s^". Ignore.\n"));e2'
     end
     else begin
       ignore(log_to_file ("LHS of assign is Id "^s^"\n") );
       (* If the rhs is object type, load it first *)
       let rhs_type = get_ast_type_of_sexpr rhs in

       (* Check AST type of expr *)
       ignore(log_to_file ("RHS has AST type "^(A.string_of_typ rhs_type)^"\n"));

       (* Decide whether to load by checking rhs type *)
       match rhs with 
         S.S_Literal(_)| S.S_BoolLit(_)| S.S_FloatLit(_)| S.S_Unop(_, _, _)| S.S_Binop(_, _, _, _)
       | S.S_Cast(_,_,_) ->
         ignore (L.build_store e2' (lookup s) llbuilder); 
         e2'
       | S.S_Call(fname, _, data_type) ->
         ignore(log_to_file ("RHS of assign is function call of type "^(A.string_of_typ data_type)^".\n"));
         let _val = L.build_store e2' (lookup s) llbuilder in
         ignore(log_to_file ("Load the value makes: "^(L.string_of_llvalue _val)^"\n"));
         _val
       | S.S_Id(name, data_type) -> (
           match data_type with 
             A.Object(name) -> 
             ignore(L.build_alloca (ltype_of_typ data_type) name llbuilder);
             ignore (L.build_store e2' (lookup s) llbuilder); 
             e2'
           | _ ->
             ignore (L.build_store e2' (lookup s) llbuilder); 
             e2'
         )
       | S.S_StringLit(_) | S.S_ArrayCreate(_, _) -> 
         ignore (L.build_store e2' (lookup s) llbuilder); 
         e2'
       | S.S_ObjAccess(obj,field,data_type) ->
         ignore(log_to_file ("RHS of assign is object access: "^(S.string_of_s_expr rhs)^"\n"));
         (* Construct function name to see if it exists *)
         let field_name_trans sexpr = match sexpr with
             S.S_Call(f, el,t) -> f 
           | S.S_Id(s,t) -> s
         in
         let class_of_obj sexpr = 
           ignore(log_to_file ("Matching sexpr "^(S.string_of_s_expr sexpr)^"\n"));
           match sexpr with 
             S.S_Id(_,t) -> A.string_of_typ t
           | S.S_ObjAccess(_,_,t) -> A.string_of_typ t
         in
         let obj_type = class_of_obj obj in
         let pseudo_func_name = obj_type^"."^(field_name_trans field) in
         ignore(log_to_file ("Accessing "^pseudo_func_name^"\n"));

         (* If rhs is function call, do not load *)
         let obj_field =  (match (L.lookup_function pseudo_func_name the_module) with
               None 	-> 
               (* The field is an id. Load it. *)
               ignore(log_to_file ("Not a function. Load as id: "^(L.string_of_llvalue e2')^"\n"));
               let _val = L.build_load e2' "tmp" llbuilder in
               ignore(log_to_file ("Loaded: "^(L.string_of_llvalue _val)^"\n"));
               ignore (L.build_store _val (lookup s) llbuilder);
               _val
             | Some f 	-> 
               (* The field is a function. Do not load it. *)
               ignore(log_to_file "A function. Do not load it.\n");
               ignore (L.build_store e2' (lookup s) llbuilder); 
               e2'
           )
         in
         obj_field
       | _ -> 
         ignore(log_to_file "The rhs to be assigned to lhs is not literal. Load it first. \n");
         ignore(log_to_file ("RHS expr is "^(S.string_of_s_expr rhs)^"\n"));
         let _val = L.build_load e2' "tmp" llbuilder in
         ignore (L.build_store _val (lookup s) llbuilder);
         _val
     end
   | S.S_ObjAccess(the_obj,the_field,data_type) ->
     ignore(log_to_file ("Building object access in LHS of assign: type  "^(A.string_of_typ data_type)^"\n") );
     let e1' = codegen_obj_access true the_obj the_field data_type llbuilder in
     ignore(log_to_file ("Built LHS ObjAccess: "^(L.string_of_llvalue e1')^"\n"));
     let e2' = codegen_sexpr llbuilder true rhs in
     ignore(log_to_file ("Built RHS ObjAccess: "^(L.string_of_llvalue e2')^"\n"));
     let _val = L.build_store e2' e1' llbuilder in
     ignore(log_to_file ("Built store: "^(L.string_of_llvalue _val)^"\n"));
     e2' 

   | S.S_ArrayAccess(e, index, typ) -> 
     let vmemory = generate_array_access false e index llbuilder in
     let value = codegen_sexpr llbuilder false rhs in
     ignore (L.build_store value vmemory llbuilder);
     value
   | _ -> 
     let e1' = codegen_sexpr llbuilder false lhs and e2' = codegen_sexpr llbuilder false rhs in
     ignore (L.build_store e2' e1' llbuilder); e2' 
  )

(* Construct code for an expression; return its value *)
and codegen_sexpr builder isReturn (sexpr:S.s_expr) = 
  ignore(log_to_file ("Building expression: "^(S.string_of_s_expr sexpr)^"\n") );

  match sexpr with
    S.S_Literal i -> 
      L.const_int i32_t i
  | S.S_BoolLit b -> L.const_int i1_t (if b then 1 else 0) 
  | S.S_FloatLit(f)          		-> L.const_float f_t f 
  | S.S_StringLit s  ->	L.build_global_stringptr s "tmp_string" builder
  | S.S_Noexpr -> L.const_int i32_t 0
  | S.S_Null -> L.const_null i32_t
  | S.S_Id (s,data_type) -> 
    if isReturn then begin
      ignore(log_to_file ("Id "^s^" is to be returned.\n"));
      let to_ret = match data_type with
          A.Object(name) -> 
          lookup s
        |_ ->
          L.build_load (lookup s) s builder 
      in
      to_ret
    end
    else begin
      ignore(log_to_file ("The Id is "^s^"\n"));
      L.build_load (lookup s) s builder 
    end
  | S.S_ArrayCreate(typ, size) -> generate_one_d_array typ size builder 
  | S.S_ArrayAccess(arrayName, index, t) -> generate_array_access true arrayName index builder
  | S.S_Binop (e1, op, e2,_) ->
    ignore(log_to_file ("Building binop "^(S.string_of_s_expr e1)^","^(A.string_of_op op)^","^(S.string_of_s_expr e2)^"\n" ) );

    let e1' = codegen_sexpr builder false e1 
    and e2' = codegen_sexpr builder false e2 in

    ignore(log_to_file ( ("\n"^L.string_of_llvalue e1')^"\n" ) );
    ignore(log_to_file ( (L.string_of_llvalue e2')^"\n" ) );

    if (Semant.get_sexpr_type(e1) = A.String ) then
      (match op with
         A.Add -> codegen_strcat e1' e2' builder 
       | _ -> raise(Failure("unsupported string opreators"))
      )
    else if (Semant.get_sexpr_type(e1) = A.Float ) then
      (match op with
         A.Add     -> L.build_fadd 
       | A.Sub     -> L.build_fsub
       | A.Mult    -> L.build_fmul
       | A.Div     -> L.build_fdiv
       | A.Equal   -> L.build_fcmp L.Fcmp.Oeq
       | A.Neq     -> L.build_fcmp L.Fcmp.One
       | A.Less    -> L.build_fcmp L.Fcmp.Olt
       | A.Leq     -> L.build_fcmp L.Fcmp.Ole
       | A.Greater -> L.build_fcmp L.Fcmp.Ogt
       | A.Geq     -> L.build_fcmp L.Fcmp.Oge
       | _ -> raise(Failure("unsupported float opreators"))
      ) e1' e2' "tmp" builder
    else
      (match op with
         A.Add     -> L.build_add 
       | A.Sub     -> L.build_sub
       | A.Mult    -> L.build_mul
       | A.Div     -> L.build_sdiv
       | A.And     -> L.build_and
       | A.Or      -> L.build_or
       | A.Equal   -> L.build_icmp L.Icmp.Eq
       | A.Neq     -> L.build_icmp L.Icmp.Ne
       | A.Less    -> L.build_icmp L.Icmp.Slt
       | A.Leq     -> L.build_icmp L.Icmp.Sle
       | A.Greater -> L.build_icmp L.Icmp.Sgt
       | A.Geq     -> L.build_icmp L.Icmp.Sge
      ) e1' e2' "tmp" builder
  | S.S_Unop(op, e, _) ->
    ignore(log_to_file ("Building unop "^(S.string_of_s_expr e)^"\n" ) );
    let e' = codegen_sexpr builder false e in
    (match (op)  with
       A.Neg     -> 
       if (Semant.get_sexpr_type(e) = A.Float ) then
         L.build_fneg
       else
         L.build_neg
     | A.Not     -> L.build_not
    ) e' "tmp" builder

  | S.S_Assign (lhs,rhs, _) ->  
    codegen_assign lhs rhs builder
  | S.S_Cast(t1,e,t2) ->
    (
      match (t1,t2) with
        (Int,Int) | (Float,Float) | (Bool,Bool) | (String,String) -> codegen_sexpr builder false e
      | (Float,Int) ->   
        let e' = codegen_sexpr builder false e in 
        L.build_fptosi e' i32_t "float_to_int" builder
      | (Int,Float) ->   
        let e' = codegen_sexpr builder false e in
        L.build_sitofp e' f_t "int_to_float" builder
      | (String,Int) ->
        let atoi_func = func_lookup "atoi" in
        L.build_call atoi_func [| (codegen_sexpr builder false e) |] "atoi" builder              
      | (String,Float) ->
        let atof_func = func_lookup "atof" in
        L.build_call atof_func [| (codegen_sexpr builder false e) |] "atof" builder              
      | (Float,String) -> codegen_tostring e builder
      | (Int,String) -> codegen_tostring e builder
      | (Bool,String) -> codegen_tostring e builder
      | _ -> raise(Failure("cast not implemented"))
    )
  |  S.S_Call ("print",[e],_ ) -> 
    let printf_format = 
      let temp = Semant.get_sexpr_type(e) in
      (match temp with
         A.Int | A.Bool -> int_format_str
       | A.String -> string_format_str 
       | A.Float -> float_format_str
       | A.Arraytype(t) -> (match t with 
             A.Int | A.Bool -> int_format_str
           | A.String -> string_format_str 
           | A.Float -> float_format_str
         )
       | _ -> raise(Failure("print only supports primitive types") )
      ) in
    let printf_func = func_lookup "printf" in
    let _val =
      match e with 
        S.S_ObjAccess(_,_,_) -> 
        L.build_load ((codegen_sexpr builder false e)) "var_for_print" builder
      | _ -> 
        codegen_sexpr builder false e
    in
    L.build_call printf_func [| (printf_format builder) ; _val |]
      "printf" builder        

  | S.S_Call ("toString",[e],_) ->
    codegen_tostring e builder
  | S.S_Call ("strcmp",e,_) ->
    let strcmp_func = func_lookup "strcmp" in 
    let actuals = List.rev (List.map (codegen_sexpr builder false) (List.rev e)) in
    L.build_call strcmp_func (Array.of_list actuals) "strcmp" builder    
  (* Add sizeof *)
  | S.S_Call ("sizeof",[e],data_type) ->
    codegen_sizeof e data_type builder
  (* Add malloc *)
  | S.S_Call ("malloc",[e],data_type) ->
    codegen_malloc e data_type builder
  (* Add cast *)
  | S.S_Call ("cast",[e],data_type) ->
    codegen_cast e data_type builder


  | S.S_Call (fname, act, _) ->
    ignore(log_to_file ("Build function "^fname^" with expr: \n") );
    ignore(List.map (fun expr -> log_to_file ((S.string_of_s_expr expr) ^ "\n") ) act );

    let (fdef, fdecl) = H.find func_table fname in
    let actuals = List.rev (List.map (codegen_sexpr builder true) (List.rev act)) in
    let result = (match fdecl.S.s_typ with A.Void -> ""
                                         | _ -> fname ^ "_result") in
    L.build_call fdef (Array.of_list actuals) result builder
  |   S_ObjCreate(id, expr_list, data_type)	-> codegen_obj_create id expr_list data_type builder 
  |   S_ObjAccess(e1, e2, d) -> codegen_obj_access (not isReturn) e1 e2 d builder


(* sizeof *)
and codegen_sizeof sexpr data_type llbuilder =
  ignore(log_to_file ("Building sizeof with data_type:"^(A.string_of_typ data_type)^"\n") );
  ignore(log_to_file ("Expr: "^(S.string_of_s_expr sexpr )^"\n"   ) );

  let type_of_el = get_underlying_type_of_sexpr sexpr in
  let size_of_el = L.size_of type_of_el in
  L.build_bitcast size_of_el i32_t "tmp" llbuilder


and codegen_tostring e llbuilder = 
  let ll_expr = codegen_sexpr llbuilder false e in
  let t = Semant.get_sexpr_type(e) in
  let (size, sprintf_format) =
    (match (t) with
       A.Int | A.Bool ->
       (L.const_int i32_t 20, int_sprintf_str(llbuilder))
     | A.Float ->
       (L.const_int i32_t 20, float_sprintf_str(llbuilder))        
     | A.String ->
       let strlen_func = func_lookup "strlen" in
       let len =  L.build_call strlen_func [| (ll_expr); |] "strlen" llbuilder in           
       let len = L.build_add len (L.const_int i32_t 1) "tmp" llbuilder in
       (L.const_int i32_t 20, string_sprintf_str(llbuilder)) 
     | _ ->  raise (Failure ("toString() function only supports primitive types") )
    ) in
  let t = i8_t in
  let buf = L.build_array_malloc t size "to_string_buf" llbuilder in
  let buf = L.build_pointercast buf (L.pointer_type t) "to_string_buf" llbuilder in 
  let sprintf_func = func_lookup "sprintf" in
  let _ = L.build_call sprintf_func [| buf; sprintf_format ; ll_expr; |]
      "sprintf" llbuilder in
  buf     

and codegen_strcat e1' e2' llbuilder = 
  let ll_expr1 = e1' in
  let ll_expr2 = e2' in
  let strlen_func = func_lookup "strlen" in
  let strcpy_func = func_lookup "strcpy" in
  let strcat_func = func_lookup "strcat" in
  let len1 =  L.build_call strlen_func [| (ll_expr1); |] "strlen" llbuilder in  
  let len2 =  L.build_call strlen_func [| (ll_expr2); |] "strlen" llbuilder in               
  let len_new = L.build_add len1 len2 "tmp" llbuilder in
  let size = L.build_add len_new (L.const_int i32_t 1) "tmp" llbuilder in 
  let t = i8_t in
  let buf = L.build_array_malloc t size "to_string_buf" llbuilder in
  let buf = L.build_pointercast buf (L.pointer_type t) "to_string_buf" llbuilder in       
  let buf = L.build_call strcpy_func [| buf; ll_expr1; |] "strcpy" llbuilder in  
  let buf = L.build_call strcat_func [| buf; ll_expr2; |] "strcat" llbuilder in  
  buf

and codegen_malloc sexpr data_type llbuilder =
  let f = func_lookup "malloc" in
  let params = List.map (codegen_sexpr llbuilder false ) [sexpr] in
  match data_type with
  A.Void -> L.build_call f (Array.of_list params) "" llbuilder
  | 	_ ->
  let c = !counter in
  let ptr = L.build_call f (Array.of_list params) (string_of_int c) llbuilder in
  counter := c + 1;
  H.add new_obj_table c ptr;
  ptr

and codegen_cast sexpr data_type llbuilder =
  ignore(log_to_file ("Building cast with data_type:"^(A.string_of_typ data_type)^"\n") );
  ignore(log_to_file ("Expr: "^(S.string_of_s_expr sexpr )^"\n"   ) );

  let cast_malloc_to_objtype lhs currType newType llbuilder = match newType with
      A.Object(name)-> 
      let obj_type = ltype_of_typ (A.Object(name)) in 
      ignore(log_to_file ("In cast obj type is "^(L.string_of_lltype obj_type)^"\n")) ;
      L.build_pointercast lhs obj_type "tmp" llbuilder
    |   _ as t -> raise (Failure("I don't know why Im here but this is wrong"))
  in
  let t = Semant.get_sexpr_type sexpr in
  let lhs = match sexpr with
    |   S.S_Id(id, data_type) -> 
      lookup id
    | _ -> codegen_sexpr llbuilder false sexpr
  in
  cast_malloc_to_objtype lhs t data_type llbuilder


(* Need to be moved*)
and int_format_str builder = L.build_global_stringptr "%d\n" "fmt" builder 
and string_format_str builder = L.build_global_stringptr "%s\n" "fmt" builder
and float_format_str builder = L.build_global_stringptr "%f\n" "fmt" builder
and int_sprintf_str builder = L.build_global_stringptr "%d" "tostr_fmt" builder 
and string_sprintf_str builder = L.build_global_stringptr "%s" "tostr_fmt" builder 
and float_sprintf_str builder = L.build_global_stringptr "%f" "tostr_fmt" builder


and codegen_classes (classes:S.s_class_decl list) =
  ignore(log_to_file "Codegen classes\n");
  ignore(List.map (fun c -> log_to_file ("To build class "^c.S.s_cname^"\n") ) classes);

  let codegen_one_class (the_class:S.s_class_decl) =
    ignore(log_to_file ("Generating class "^the_class.s_cname^"\n") );

    (* Generate struct decl *)
    let _ = codegen_struct_stub the_class in 

    (* Generate class local vars *)
    let _ = codegen_struct the_class in  

    (* Generate functions *)
    ignore(log_to_file "Finished class definition. Now start with functions\n");
    ignore(List.map (fun f -> log_to_file ("To build constructor "^f.S.s_fname^"\n") ) the_class.s_constructors);
    ignore(List.map (fun f -> log_to_file ("To build method "^f.S.s_fname^"\n") ) the_class.s_methods);

    let _ = codegen_function_decls the_class.s_constructors in
    let _ = codegen_function_decls the_class.s_methods in
    let _ = codegen_fbody the_class.s_constructors in

    let _ = codegen_fbody the_class.s_methods in
    ignore(log_to_file ("Generated class "^the_class.s_cname^"\n") );
  in  
  List.map codegen_one_class classes;
  ignore(log_to_file "Finished generating classes\n")

and find_struct name = 
  try Hashtbl.find object_types name
  with | Not_found -> raise(Failure ("Cannot find object type "^name) )

and codegen_fbody functions =

  (* Fill in the body of the given function *)
  let build_function_body fdecl =
    let _ = log_to_file ("Building function " ^ fdecl.S.s_fname ^ "\n" ) in

    (* First clear local and param tables *)
    let _ = H.clear locals_table; H.clear params_table in 

    (* Find target function *)
    let (the_function, _) = H.find func_table fdecl.S.s_fname in
    let builder = L.builder_at_end context (L.entry_block the_function) in

    (* Construct the function's "locals": formal arguments and locally
       declared variables.  Allocate each on the stack, initialize their
       value, if appropriate, and remember their values in the "locals" map *)
    let codegen_locals local_vars = 
      let add_formal table (the_type,name) value =
        L.set_value_name name value;
        let local = match the_type with
            A.Object(name) ->
            value
          | _ ->          
            let _val = L.build_alloca (ltype_of_typ the_type) name builder in 
            ignore (L.build_store value _val builder); (* local is the pointer we store at *)
            _val
        in
        H.add table name local
      in
      let add_local table (the_type,name) =
        let t = 
          match the_type with
            A.Object(cname) ->
            find_struct cname
          | _ -> 
            ltype_of_typ the_type
        in

        let local_var = L.build_alloca t name builder
        in                               
        H.add table name local_var
      in
      let _ = List.map2 (add_formal params_table) fdecl.S.s_formals (Array.to_list (L.params the_function)) 
      in
      ignore(List.map (add_local locals_table) local_vars);

      (* Log to file all the locals and formals *)
      ignore(log_to_file "Logging local table\n");
      print_hashtable locals_table
    in
    codegen_locals fdecl.S.s_locals;

    (* Invoke "f builder" if the current block doesn't already
       have a terminal (e.g., a branch). *)
    let add_terminal builder f =
      match L.block_terminator (L.insertion_block builder) with
        Some _ -> ()
      | None -> ignore (f builder) 
    in
    (* Build the code for the given statement; return the builder for
       the statement's successor *)
    let rec codegen_stmt builder stmt = 
      match stmt with
        S.S_Block sl -> 
        List.fold_left codegen_stmt builder sl
      | S.S_Expr (e,_) -> 
        ignore (codegen_sexpr builder false e); builder
      | S.S_Return (e,_) -> 
        ignore(log_to_file ("Building return expr "^(S.string_of_s_expr e)^"\n" ) );
        ignore (match fdecl.S.s_typ with
              A.Void -> L.build_ret_void builder
            | _ -> L.build_ret (codegen_sexpr builder true e) builder); builder
      | S.S_If (predicate, then_stmt, else_stmt) ->
        let bool_val = codegen_sexpr builder false predicate in
        let merge_bb = L.append_block context "merge" the_function in

        let then_bb = L.append_block context "then" the_function in
        add_terminal (codegen_stmt (L.builder_at_end context then_bb) then_stmt)
          (L.build_br merge_bb);

        let else_bb = L.append_block context "else" the_function in
        add_terminal (codegen_stmt (L.builder_at_end context else_bb) else_stmt)
          (L.build_br merge_bb);

        ignore (L.build_cond_br bool_val then_bb else_bb builder);
        L.builder_at_end context merge_bb

      | S.S_While (predicate, body) ->
        let pred_bb = L.append_block context "while" the_function in
        ignore (L.build_br pred_bb builder);

        let body_bb = L.append_block context "while_body" the_function in
        add_terminal (codegen_stmt (L.builder_at_end context body_bb) body)
          (L.build_br pred_bb);

        let pred_builder = L.builder_at_end context pred_bb in
        let bool_val = codegen_sexpr pred_builder false predicate in

        let merge_bb = L.append_block context "merge" the_function in
        ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder);
        L.builder_at_end context merge_bb

      | S.S_For (e1, e2, e3, body) -> codegen_stmt builder
                                        ( S.S_Block [S.S_Expr(e1,Semant.get_sexpr_type(e1)) ;
                                                     S.S_While (e2, S.S_Block [body ; S.S_Expr(e3,Semant.get_sexpr_type(e3))]) ] )
    in
    (* let ff = func_lookup "free" in
    let counters = H.to_seq_values new_obj_table in 
  (* each element in the table -> llvalue, which is a pointer*)
    let _ = Format.print_string fdecl.S.s_fname in
    let _ = if fdecl.S.s_fname == "main" then 
    (Seq.iter (fun c -> L.build_call ff (Array.of_list [c]) "" builder; () ) counters)
    else () in *)
    (* Build the code for each statement in the function *)
    let builder_stmt_built = codegen_stmt builder (S.S_Block fdecl.S.s_body) 
    in
    (* Add a return if the last block falls off the end *)
    add_terminal builder_stmt_built (match fdecl.S.s_typ with
          A.Void -> L.build_ret_void
        | t -> L.build_ret (L.const_int (ltype_of_typ t) 0))
  in
  List.iter build_function_body functions
    
let codegen_sprogram (program:S.s_program) =
  (* First wipe out log file *)


  (* Build builtin function declarations *)
  let _ = codegen_builtins program.S.builtins in

  let _ = codegen_globals program.S.global_vars in 

  let _ = codegen_classes program.S.classes in 

  (* Generate functions here *)
  let _ = codegen_function_decls program.S.functions in 
  let _ = codegen_function_decls (program.S.main::[]) in  
  let _ = codegen_fbody program.S.functions in 
  let _ = codegen_fbody (program.S.main::[]) in 
  the_module