(* Abstract Syntax Tree and functions for printing it *)

type op = Add | Sub | Mult | Div | Equal | Neq | Less | Leq | Greater | Geq |
          And | Or

type uop = Neg | Not

type typ = Int | Bool | Void | Float | Null | String | Object of string | Arraytype of typ | Constructortyp | Any

type bind = typ * string

type expr =
    Literal of int
  | BoolLit of bool 
  | StringLit of string
  | FloatLit of float
  | Id of string
  | Binop of expr * op * expr
  | Unop of uop * expr
  | Assign of expr * expr
  | Call of string * expr list
  | ObjAccess of expr * expr
  | ObjCreate of string * expr list
  | ArrayCreate of typ * int
  | ArrayAccess of expr * int
  | Delete of expr
  | Noexpr
  | Nullexpr
  | Cast of typ * expr

type stmt =
    Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt

type func_decl = {
    typ : typ;
    fname : string;
    formals : bind list;
    locals : bind list;
    body : stmt list;
  }

type cbody = {
    fields: bind list;
    methods: func_decl list;
    constructors: func_decl list;
}

type class_decl = {
    cname: string;
    cbody: cbody;
}

type program = bind list * func_decl list * class_decl list

(* Pretty-printing functions *)
let rec string_of_typ = function
    Int -> "int"
  | Bool -> "bool"
  | Void -> "void"
  | String -> "string"
  | Float -> "float"
  | Null -> "null"
  | Object(cname) -> cname
  | Arraytype(t) -> "array of " ^ string_of_typ t
  | Constructortyp -> "constructortyp"
  | Any -> "any"

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "&&"
  | Or -> "||"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"

let rec string_of_expr = function
    Literal(l) -> string_of_int l
  | BoolLit(true) -> "true"
  | BoolLit(false) -> "false" 
  | StringLit(s) -> s
  | FloatLit(f) -> string_of_float f
  | Id(s) -> s
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  | Unop(o, e) -> string_of_uop o ^ string_of_expr e
  | Assign(e1, e2) -> string_of_expr e1 ^ " = " ^ string_of_expr e2
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | ObjCreate(s,el) -> "new " ^ s ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | ObjAccess(e1,e2) -> string_of_expr e1 ^ "." ^ string_of_expr e2
  | ArrayCreate(typ, len) -> "(array of type " ^ string_of_typ typ ^ " with length " ^ string_of_int len ^ ")"
  | ArrayAccess(arrayName, index) -> "(array name: " ^ string_of_expr arrayName ^ " index: " ^ string_of_int index ^ ")"
  | Delete(e) -> "delete" ^ string_of_expr e
  | Cast (t,e) -> "(" ^ (string_of_typ t) ^ ") " ^ (string_of_expr e)
  | Noexpr -> ""
  | Nullexpr -> "null"

let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let string_of_vdecl (t, id) = string_of_typ t ^ " " ^ id ^ ";\n"

let string_of_formal (t, id) = string_of_typ t ^ " " ^ id

let string_of_fdecl fdecl =
  string_of_typ fdecl.typ ^ " " ^
  fdecl.fname ^ "(" ^ String.concat ", " (List.map string_of_formal fdecl.formals) ^
  ")\n{\n" ^
  String.concat "" (List.map string_of_vdecl fdecl.locals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_cdecl cdecl = 
    "class" ^ " " ^ cdecl.cname ^ " { \n" ^ 
    String.concat "" (List.map string_of_vdecl cdecl.cbody.fields) ^
    String.concat "" (List.map string_of_fdecl cdecl.cbody.constructors) ^
    String.concat "" (List.map string_of_fdecl cdecl.cbody.methods) ^
    "}\n"


let string_of_program (vars, funcs, classes) =
  String.concat "" (List.map string_of_vdecl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_cdecl classes) ^ "\n" ^ 
  String.concat "\n" (List.map string_of_fdecl funcs)
