(* ARBOL Scanner *)
{ open Parser }

let digit = ['0'-'9']
let identifier = ['a'-'z''_']+['a'-'z''A'-'Z''_''0'-'9']*
 
rule token = parse
  	(* Whitespace *)
  	| [' ' '\t' '\n'] { token lexbuf }
  	(* Comments *)
	| "#" { comment lexbuf }
	| "#*" { comment_multi_line lexbuf }

 	(* Punctuation *)
 	(* Check for - comma, period  - usage *)
	| '(' {LPAREN}			| ')' {RPAREN}
	| '{' {LBRACE}  		| '}' {RBRACE}
	| ';' {SEMI} 			
	| ',' {COMMA}	
	| '~' {DEREF}			
	| '@' {NODE_INIT}			
	| "[" {GET_CHILD_L}
	| "]" {GET_CHILD_R}
	| "=>" {NODE_ASSIGN}			
	| "<--" {LCHILD}
	| "-->"	{RCHILD}

	(* Function Keywords *)
	| "function"	{FUNCTION}
	| "return"		{RETURN}

	(* Conditionals and Loops Keywords *)
	| "if" 			{IF}
	| "else" 		{ELSE}
	| "for" 		{FOR}
	| "while"		{WHILE}
	| "continue"	{CONTINUE}
	| "break"		{BREAK}

	(* Types *)
	| "int" 		{INT}
	| "float" 		{FLOAT}
	| "boolean" 	{BOOL}
	| "char" 		{CHAR}
	| "void"		{VOID}
	| "string"		{STRING}


	(* Arithmetic Operators *)
	| '+' {PLUS}		| '-' { MINUS }
	| '%' {MOD}			| '*' { TIMES }
	| '/' { DIVIDE }

	(* Assignment Operators *)
	| "="	{ASSIGN}
	(*| "--" {MINUS_MINUS}
	| "++" {PLUS_PLUS}*)

	(* Relational Operators *)
	| "=="	{EQUALS}		| "!="	{NOT_EQUALS}
	| '>'	{GREATER_THAN}	| ">="	{GREATER_THAN_EQUALS}
	| '<'	{LESS_THAN}		| "<="	{LESS_THAN_EQUALS}

	(* Logical Operators *)
	| "&&" {AND}
	| "||" {OR}
	| "!"  {NOT}

	(* Literals *)
	| "true"	{ BOOL_LIT(true)}
	| "false"	{ BOOL_LIT(false)}
	| identifier as lit {ID(lit)}
	| digit+ as lit {INT_LIT(int_of_string lit)}
	| ('-'? digit+ '.' digit* | '-'? '.' digit+) as lit {FLOAT_LIT(lit)}
	| '"' + [',''_''%''^''/''['']''('')''*''&''$''#''@''!''=''+''-'' ' '\'' '0'-'9''a'-'z''A'-'Z''\\']* + '"' as lit {
		let strip_quotes str = String.sub str 1 ((String.length str) - 2) in STRING_LIT(strip_quotes lit)
		}
	| '\'' ([^'\''] as lit) '\'' {CHAR_LIT(lit)}
	| "\'\\n\'" {CHAR_LIT('\n')}
	| "\'\\t\'" {CHAR_LIT('\t')}
	| "\'\\\\\'" {CHAR_LIT('\\')}

	(* End of File *)
	| eof {EOF}

	(*Illegal characters*)
	| _ as c { raise (Failure("Illegal character: " ^ Char.escaped c))}

and comment = parse
	| '\n' {token lexbuf}
	| eof  {EOF}
	| _	   {comment lexbuf}
and comment_multi_line = parse
	| "*#"	{token lexbuf}
	| _ 	{comment_multi_line lexbuf}
	
