(* ARBOL Semantically Checked abstract syntax tree *)

open Ast

(* Expressions *)
(* type sexpr = var_type * expr_detail Expressions that have types *)
type sexpr = 
	  SInt_lit of int
	| SFloat_lit of string
	| SBool_lit of bool
	| SChar_lit of char
	| SString_lit of string
	| SId of string * vtype
	| SAssign of string * sexpr * vtype
	| SUnop of unop * sexpr * vtype
	| SBinop of sexpr * op * sexpr * vtype * vtype * vtype
	| SCall of string * sexpr list * vtype
	| SNodeop of node_op * string * vtype
	| SNode_assign of string * sexpr * vtype
	| SNoexpr

type svdecl = {
	sv_type: vtype;
	sv_name: string;
	sv_val: sexpr;
	sv_node_val: bool;
}

(* Statements *)
type sstmt = 
	  SBlock of sstmt list
	| SExpr of sexpr
	| SReturn of sexpr * vtype
	| SIf of sexpr * sstmt * sstmt 
	| SFor of sexpr * sexpr * sexpr * sstmt
	| SWhile of sexpr * sstmt 
	| SContinue 
	| SBreak
	| SNode_child of string * node_child_op * sexpr * vtype
	| SVariable of svdecl 

(* Function Declaration *)
type sfdecl = {
	srtype: vtype;
	sfname: string;
	sargs_list: (vtype * string) list; (* formal arguments*)
	sbody: sstmt list;
}

type sprogram = svdecl list * sfdecl list

let rec string_of_sexpr = function
      SInt_lit(l) -> string_of_int l
    | SFloat_lit(l) -> l
    | SBool_lit(true) -> "true"
    | SBool_lit(false) -> "false"
    | SChar_lit(l) -> String.make 1 l
    | SString_lit(l) -> l
    | SId(s, _) -> s
    | SAssign(v, e, _) -> v ^ " = " ^ string_of_sexpr e
    | SUnop(o, e, _) -> string_of_unop o ^ string_of_sexpr e
    | SBinop(e1, o, e2, _, _, _) -> 
				string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
		| SCall(f, el, _) -> f ^ "(" ^ String.concat ", " (List.map string_of_sexpr el) ^ ")"
    | SNodeop(o, n1, _) -> string_of_node_op o ^ n1
    | SNode_assign(n1, e, _) -> n1 ^ " " ^ string_of_node_assign_op ^ " " ^ string_of_sexpr e
		| SNoexpr -> ""


let string_of_svdecl var = 
	match var.sv_val with 
  | SNoexpr -> string_of_vtype var.sv_type ^ " " ^ var.sv_name 
	| _ ->
	match var.sv_node_val with
	| true -> string_of_vtype var.sv_type ^ " " ^ var.sv_name ^ " " ^ string_of_node_assign_op ^ " " ^ string_of_sexpr var.sv_val
	| false -> string_of_vtype var.sv_type ^ " " ^ var.sv_name ^ " = " ^ string_of_sexpr var.sv_val

let string_of_sfdecl(v_type, name) = string_of_vtype v_type ^ " " ^ name

let rec string_of_sstmt = function 
	  SBlock(stmts) ->
	  	"{\n" ^ String.concat "" (List.map string_of_sstmt stmts) ^ "}\n"
	| SExpr(expr) -> string_of_sexpr expr ^ ";\n";
	| SReturn(expr, _) -> "return " ^ string_of_sexpr expr ^ ";\n";
	| SIf(e, s, SBlock([])) -> "if (" ^ string_of_sexpr e ^ ")\n" ^ string_of_sstmt s
	| SIf(e, s1, s2) ->  "if (" ^ string_of_sexpr e ^ ")\n" ^
	  string_of_sstmt s1 ^ "else\n" ^ string_of_sstmt s2
	| SFor(e1, e2, e3, s) ->
	  "for (" ^ string_of_sexpr e1  ^ " ; " ^ string_of_sexpr e2 ^ " ; " ^
	  string_of_sexpr e3  ^ ") " ^ string_of_sstmt s
	| SWhile(e, s) -> "while (" ^ string_of_sexpr e ^ ") " ^ string_of_sstmt s
	| SContinue -> "continue" ^ ";\n"
	| SBreak -> "break" ^ ";\n"
	| SNode_child(n1, o, e, _) -> n1 ^ " " ^ string_of_node_child_op o ^ " " ^ string_of_sexpr e ^ ";\n"
	| SVariable(v) -> string_of_svdecl v ^ ";\n"

let string_of_sfdecl fdecl =
  "function " ^ string_of_vtype fdecl.srtype ^ " " ^
  fdecl.sfname ^ "(" ^ 
	String.concat ", " (List.map string_of_sfdecl fdecl.sargs_list) ^ 
  ")\n{\n" ^
  String.concat "" (List.map string_of_sstmt fdecl.sbody) ^
  "}\n"

let string_of_program (vlist, flist) =
	"START\n" ^ 
	(String.concat "\n" (List.map string_of_svdecl vlist)) ^ "\n\n" ^
  	(String.concat "\n\n" (List.map string_of_sfdecl (List.rev flist) )) ^ 
	"\nEND\n"
