(* ARBOL Abstract Syntax Tree *)

(* Binary operators *)
type op = Plus  | Minus | Times | Divide | Mod 
		| Equals | Not_Equals | Greater | Greater_Eq 
		| Less  | Less_Eq | And | Or

(* Unary operators *)
type unop = Neg | Not 

(* Post fix operators *)

(* Tree (Node) operators *)

type node_op = Get_left_child | Get_right_child | Dref

type node_child_op = Set_left_child | Set_right_child

type node_assign_op = Node_assign

(* Assignment Operators *)
type vtype = 
  | Int
	| Float
	| Bool
	| Char
	| Void
	| String
	| Node of vtype

(* Expressions *)
type expr = 
	  Int_lit of int
	| Float_lit of string
	| Bool_lit of bool
	| Char_lit of char
	| String_lit of string
	| Id of string
	| Assign of string * expr 
	| Unop of unop * expr
	| Binop of expr * op * expr
	| Call of string * expr list
	| Nodeop of node_op * string
	| Node_assign of string * expr
	| Noexpr
	(* 	
	| Not of expr 
	| Array of expr list;
	| ArrayAccess of string * expr
	| Value of expr * expr
	*)

(* Variable Declaration *)

type vdecl = {
	v_type: vtype;
	v_name: string;
	v_val: expr;
	node_val: bool;
}

(* Statements *)
type stmt = 
	  Block of stmt list
	| Expr of expr
	| Return of expr
	| If of expr * stmt * stmt
	| For of expr * expr * expr * stmt
	| While of expr * stmt 
	| Continue 
	| Break
	| Node_child of string * node_child_op * expr
	(* | ArrayAssign of string * expr * expr *)
	| Variable of vdecl 

(* Function Declaration *)
type fdecl = {
	rtype: vtype;
	fname: string;
	args_list: vdecl list; (* formal arguments*)
	body: stmt list;
}

(* type program = vdecl * func_decl *)
type program = vdecl list * fdecl list

(* Need this? *)
exception Variable_of_void

(*
 * Prints out Abstract Syntax Tree
 *)

let string_of_op = function 
	  Plus   -> "+"
	| Minus  -> "-"
	| Times  -> "*"
	| Divide -> "/"
	| Mod    -> "%"
	| Equals  -> "=="
	| Not_Equals -> "!="
	| Less -> "<"
	| Less_Eq -> "<="
	| Greater -> ">"
	| Greater_Eq -> ">="
	| And -> "&&"
	| Or -> "||"

let string_of_unop = function
	| Neg -> "-"
	| Not -> "!"

let string_of_node_op = function
	| Get_left_child -> "<^"
	| Get_right_child -> "^>"
	| Dref -> "~"

let string_of_node_child_op = function
	| Set_right_child -> "-->"
	| Set_left_child -> "<--"

let string_of_node_assign_op = "=>"

let rec string_of_expr = function
      Int_lit(l) -> string_of_int l
    | Float_lit(l) -> l
    | Bool_lit(true) -> "true"
    | Bool_lit(false) -> "false"
    | Char_lit(l) -> String.make 1 l
    | String_lit(l) -> l
    | Id(s) -> s
    | Assign(v, e) -> v ^ " = " ^ string_of_expr e
    | Unop(o, e) -> string_of_unop o ^ string_of_expr e
    | Binop(e1, o, e2) -> 
        string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
    | Call(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
    | Nodeop(o, n1) -> string_of_node_op o ^ n1
    | Node_assign(n1, e) -> n1 ^ " " ^ string_of_node_assign_op ^ " " ^ string_of_expr e
    | Noexpr -> ""

let rec string_of_vtype = function
	| Int -> "int"
	| Bool -> "bool"
  | Float -> "float"
  | Char -> "char"
  | String -> "string"
  | Void -> "void"
  | Node(x) -> string_of_vtype x ^ " @"

let string_of_vdecl var = 
	match var.v_val with 
  | Noexpr -> string_of_vtype var.v_type ^ " " ^ var.v_name 
	| _ ->
	match var.node_val with
	| true -> string_of_vtype var.v_type ^ " " ^ var.v_name ^ " " ^ string_of_node_assign_op ^ " " ^ string_of_expr var.v_val
	| false -> string_of_vtype var.v_type ^ " " ^ var.v_name ^ " = " ^ string_of_expr var.v_val

let string_of_fdecl var = string_of_vtype var.v_type ^ " " ^ var.v_name

let rec string_of_stmt = function 
	  Block(stmts) ->
	  	"{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
	| Expr(expr) -> string_of_expr expr ^ ";\n";
	| Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
	| If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
	| If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
	  string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
	| For(e1, e2, e3, s) ->
	  "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
	  string_of_expr e3  ^ ") " ^ string_of_stmt s
	| While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
	| Continue -> "continue" ^ ";\n"
	| Break -> "break" ^ ";\n"
	| Node_child(n1, o, e) -> n1 ^ " " ^ string_of_node_child_op o ^ " " ^ string_of_expr e ^ ";\n"
	| Variable(v) -> string_of_vdecl v ^ ";\n"

let string_of_fdecl fdecl =
  "function " ^ string_of_vtype fdecl.rtype ^ " " ^
  fdecl.fname ^ "(" ^ 
	String.concat ", " (List.map string_of_fdecl fdecl.args_list) ^ 
  ")\n{\n" ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_program (vlist, flist) =
	"START\n" ^ 
		(String.concat "\n" (List.map string_of_vdecl vlist)) ^ "\n\n" ^
  	(String.concat "\n\n" (List.map string_of_fdecl (List.rev flist) )) ^ 
	"\nEND\n"