import click, os, sys, subprocess
import json


def get_fixtures():
    with open('arbol_test/fixtures.json') as f:
        expected_values = json.load(f)
    return expected_values


TESTS = os.listdir('tests/')
EXPECTED = get_fixtures()


@click.command()
@click.option('--test', '-t', required=False)
def cli(test):
    display_header()

    if test:
        run_test_file(test)
    else:
        for f in TESTS:
            if f.endswith('.arbol'):
                run_test_file(f)


def display_header():
    click.echo(
        click.style(
            '====================== ARBOL Tests ======================', 
            fg='blue', 
            bold=True
        )
    )


def run_test_file(test_file):
    test_name = test_file[:-6]

    cmd = """
       ./arbol -c ./tests/{test_file} > {test_name}.ll
        /usr/local/opt/llvm@11/bin/llc -relocation-model=pic {test_name}.ll > {test_name}.s
         /usr/bin/cc -c -DBUILD_TEST print.c
         /usr/bin/cc -o {test_name}.exe {test_name}.s print.o
         ./{test_name}.exe > {test_name}.out
    """.format(test_file=test_file, test_name=test_name)

    p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    outs = str(p.stdout.readline().decode('utf-8'))

    if not outs:
        test_out_file = "{test_name}.out".format(test_name=test_name)
        if os.path.isfile(test_out_file):
            assertEquals(test_out_file, EXPECTED.get(test_out_file))
    else:
        print('`%s` %s' % (test_name, outs), end = ' ')
        click.echo(click.style('FAIL', fg='red'))    


def assertEquals(output_file, expected):
    result = list()
    
    with open(output_file) as f:
        for line in f:
            result.append(line.rstrip())
    
    print('`%s` Actual: %s | Expected: %s' % (output_file[:-4], result, expected), end = ' ')
    
    if result == expected:
        click.echo(click.style('PASS', fg='green'))
    else:
        click.echo(click.style('FAIL', fg='red'))
