(* Semantic checking for the FFBB compiler *)

open Ast
open Sast

module StringMap = Map.Make (String)

(*Add variable name(key) and variable type(value) to StringMap*)
let add_to_ctxt v_type v_name ctxt =
  let ctxt = StringMap.add v_name v_type ctxt in 
  ctxt

(* Returns the type (value) given variable name (key) in StringMap *)
let find_in_ctxt v_name ctxt =
  try
    StringMap.find v_name ctxt
  with Not_found -> raise (Failure ("undeclared reference " ^ v_name))
(* Semantic checking of the AST. Returns an SAST if successful,
   throws an exception if something is wrong.

   Check each global variable, then check each function *)
let rec get_decl decls column_name = 
  (match decls with
  | [] -> raise (Failure ("Column does not exist"))
  | x::xs -> 
    if ((snd x)=column_name) then x
    else get_decl xs column_name
  )


let check (s: Ast.stmt) =
    let check_cell ctxt = function
    | IntLit(x) -> ((Int, SIntLit(x)), ctxt)
    | StrLit(x) -> ((String, SStrLit(x)), ctxt)
    | BoolLit(x) -> ((Bool, SBoolLit(x)), ctxt)
    | FloatLit(x) -> ((Float, SFloatLit(x)), ctxt)
    in

    let rec check_tableexpr ctxt = function
    | TableLit(n) -> ((find_in_ctxt n ctxt, STableLit(n)), ctxt);
    
    | Create(decls) -> ((Table decls, SCreate(decls)), ctxt)
    | Insert(old_table, exprs) -> 
      let ((table_struct, table_value), ctxt) = check_tableexpr ctxt old_table in
      let se = List.map fst (List.map (check_expr ctxt) exprs) in
      (match table_struct with
        | Table decls -> 
          let typ = List.map fst decls 
          and typ' = List.map fst se
          in ( match typ = typ' with
          | true -> 
            let value = SInsert((table_struct,table_value), se) in
            ((table_struct, value), ctxt)
          | _-> raise (Failure ("Incompatible types in Insert"))
          )
        | _ -> raise (Failure ("Insert only works for table"))
      )
      
    | Where(old_table, (c, op, s)) -> 
      let ((table_struct, table_value), ctxt) = check_tableexpr ctxt old_table in
      (
        match table_struct with
        | Table decls -> 
          let col = get_decl decls c in
          let sc = fst (check_expr ctxt s) in
          (
            (match (fst col = fst sc) with
            true -> 
              let value = SWhere((table_struct, table_value), (col, op, sc)) in
              ((Table decls, value), ctxt)
            | _ -> raise (Failure ("Incompatible types in Where")))
          )
          
        | _ -> raise (Failure ("Incompatible types in Select"))
      )
    | Delete(old_table, (c, op, s)) -> 
      let ((table_struct, table_value), ctxt) = check_tableexpr ctxt old_table in
      (
        match table_struct with
        | Table decls -> 
          let col = get_decl decls c in
          let sc = fst (check_expr ctxt s) in
          (
            (match (fst col = fst sc) with
            true -> 
              let value = SDelete((table_struct, table_value), (col, op, sc)) in
              ((Table decls, value), ctxt)
            | _ -> raise (Failure ("Incompatible types in Where")))
          )
          
        | _ -> raise (Failure ("Incompatible types in Select"))
      )
    
    | Select(old_table, selected_columns) -> 
      let ((table_struct, table_value), ctxt) = check_tableexpr ctxt old_table in
      (
        match table_struct with
        | Table decls -> 
          let new_columns = List.rev (List.map (get_decl decls) selected_columns) in
          let value = SSelect((table_struct, table_value), new_columns) in
          ((Table new_columns, value), ctxt)
        | _ -> raise (Failure ("Incompatible types in Select"))
      )
    | Distinct(old_table, selected_columns) -> 
      let ((table_struct, table_value), ctxt) = check_tableexpr ctxt old_table in
      (
        match table_struct with
        | Table decls -> 
          let new_columns = List.rev (List.map (get_decl decls) selected_columns) in
          let value = SDistinct((table_struct, table_value), new_columns) in
          ((Table new_columns, value), ctxt)
        | _ -> raise (Failure ("Incompatible types in Select"))
      )
    | ReadFile(f, pmtv) -> 
      let chan = open_in f in
      let line = input_line chan in
      close_in chan;
      let headers = List.rev (String.split_on_char ',' line) in
      let decls = List.combine pmtv headers in
      ((Table(decls), SReadFile(f, pmtv)), ctxt)
    | _ -> raise (Failure ("Table thing"))

    (* Return a semantically-checked expression, i.e., with a type *)
    and check_expr ctxt = function
    | TableExpr(t) -> 
      let((typ, v), ctxt) = check_tableexpr ctxt t in
      ((typ, STableExpr((typ, v))), ctxt)
    | Binop(e1, op, e2) -> 
      let ((lt, le), ctxt) = check_expr ctxt e1 in 
      let ((rt, re), ctxt) = check_expr ctxt e2 in
      let sbinop = (SBinop((lt, le), op, (rt, re)), ctxt) in
      (match op with
        | Add | Sub | Mul | Div when lt = Int && rt = Int -> ((Int, fst sbinop), ctxt)
        | Add | Sub | Mul | Div when lt = Float && rt = Float -> ((Float, fst sbinop), ctxt)
        (*string concate*)
        | Add when lt = String && rt = String -> ((String, fst sbinop),ctxt)
        (* equ an neq works as soon as two arguments have the same type*)
        | Equ | Neq when lt = rt -> ((Bool, fst sbinop), ctxt)
        | And | Or when lt = Bool && rt = Bool -> ((Bool, fst sbinop), ctxt)
        | Lt | Gt | Gteq | Lteq when (lt = Int && rt = Int) || (lt = Float && rt = Float) -> ((Bool, fst sbinop), ctxt)
        | _ -> raise (Failure ("Incompatible types in Binop"))
      )
    | Val(n) -> ((find_in_ctxt n ctxt, SVal(n)), ctxt)
    | Assign(t, e1, e2) ->(
      match t with
      | Table decl1 -> (
        let ((t2, se2),ctxt) = check_expr ctxt e2 in 
          (match t2 with
          | Table decl2 -> 
            let pmtv1 = List.map fst decl1
            and pmtv2 = List.map fst decl2 in
              (match (pmtv1 = pmtv2) with
              true -> let ctxt = add_to_ctxt t2 e1 ctxt in ((t, SAssign(t, e1, (t2, se2))), ctxt)
              | _ -> raise (Failure ("Incompatible types in assign table1"))
              )
          | _ -> raise (Failure ("Incompatible types in assign table2"))
          )
      )
      | _ ->
        (let ((t2, se2),ctxt) = check_expr ctxt e2 in 
          (match (t = t2) with
          true -> let ctxt = add_to_ctxt t e1 ctxt in ((t, SAssign(t, e1, (t2, se2))), ctxt)
          | _ -> raise (Failure ("Incompatible types in assign"))
          )
        )
    )
    | Reassign(v, e) ->
      let ((t, se), ctxt) = check_expr ctxt e in
      let t' = find_in_ctxt v ctxt in
      (match (t' = t) with
      true -> let ctxt = add_to_ctxt t v ctxt in ((t, SReassign(v, (t, se))), ctxt)
      | _ -> raise (Failure ("Incompatible types in Reassign")))
    | Not x -> 
      let ((t, e), ctxt) = check_expr ctxt x in 
      let v = 
      (match t with
      | Bool -> (Bool, SNot((t, e)))
      | _ -> raise (Failure ("Incompatible types in Not"))) in
      (v, ctxt)

    | Cell x -> 
        let ((typ, v), ctxt) = (check_cell ctxt x) in
        ((typ, SCell(v)), ctxt)
    | _ -> raise (Failure ("TO DO"));
    in


    (* Return a semantically-checked statement i.e. containing sexprs *)
    let rec check_stmt ctxt = function
      Expr e -> 
        let (r, ctxt) = (check_expr ctxt e) in
        (SExpr (r), ctxt)
      | Print(e) -> 
        let (r, ctxt) = (check_expr ctxt e) in
        (SPrint (r), ctxt)
      | Semi(e,e1) -> 
        let (r1, ctxt) = (check_stmt ctxt e) in
        let (r2, ctxt) = (check_stmt ctxt e1) in
        (SSemi (r1, r2), ctxt)
      | Semi1(e) -> 
        let (r, ctxt) = (check_stmt ctxt e) in
        (SSemi1(r), ctxt) 
      | Condition(c, e1) -> 
        let (r1, ctxt) = (check_expr ctxt c) in
        let (r2, ctxt) = (check_stmt ctxt e1) in
        let (r1_t, _) = r1 in 
        (match r1_t with
          | Bool -> (SCondition(r1, r2), ctxt)
          | _ -> raise (Failure ("If-else only accepts boolean value"))
        )
      | ConditionWithElse(c, e1, e2) -> 
        let (r1, ctxt) = (check_expr ctxt c) in
        let (r2, ctxt) = (check_stmt ctxt e1) in
        let (r3, ctxt) = (check_stmt ctxt e2) in
        let (r1_t, _) = r1 in 
        (match r1_t with
          | Bool -> (SConditionWithElse(r1, r2, r3), ctxt)
          | _ -> raise (Failure ("If-else only accepts boolean value"))
        )
        
      | While(e,s) ->
        let (r1, ctxt) = (check_expr ctxt e) in
        let (r2, ctxt) = (check_stmt ctxt s) in
        let (r1_t, _) = r1 in 
        (match r1_t with
          | Bool -> (SWhile (r1, r2) , ctxt)
          | _ -> raise (Failure ("If-else only accepts boolean value"))
        )
      | _-> raise (Failure ("TO DO"));

    in
    (* body of check_function *)
    fst(check_stmt StringMap.empty s)
