(* Ocamllex scanner for MicroC *)

{ open Parser }

rule tokenize = parse
  [' ' '\t' '\r' '\n'] { tokenize lexbuf }
| "/*"     { comment lexbuf }  
| '+' { PLUS }
| '-' { MINUS }
| '*' { TIMES }
| '/' { DIVIDE }
| "==" { EQUAL }
| "!=" { NOTEQUAL }
| "&&" { AND }
| ';' { SEMICOLON }
| '<' { LT }
| '>' { GT }
| ">=" { GTEQUAL }
| "<=" { LTEQUAL }
| "NOT" { NOT }
| '=' { ASSIGN }
| "||" { OR }
| "if" { IF }
| "else" { ELSE }
| "while" { WHILE }
| "return" { RETURN }
| "int" { INT }
| "string" { STRING }
| "float" { FLOAT }
| "boolean" { BOOL }
| "TABLE" { TABLE }
| "COLUMN" { COLUMN }
| "inner" { INNER }
| "left" { LEFT }
| "right" { RIGHT }
| ".DELETE" { DELETE }
| ".INSERT" { INSERT }
| ".JOIN" { JOIN }
| ".SELECT" { SELECT }
| ".EXTEND" { EXTEND }
| ".DISTINCT" { DISTINCT }
| ".WHERE" { WHERE }
| "IMPORT" { IMPORT }
| "print" { PRINT }
| "AS" { AS }
| '{' { LBRACKET }
| '}' { RBRACKET }
| '(' { LPARENTHESIS }
| ')' { RPARENTHESIS }
| ',' { COMMA }
| ['0'-'9']+ as lit { INTLIT(int_of_string lit) }
| ['0'-'9']*"."['0'-'9']+ as lxm { FLOATLIT(float_of_string lxm) }
| "true"|"false" as lxm { BOOLLIT(bool_of_string lxm) }
| '"' { str (Buffer.create 16) lexbuf }
| ['a'-'z']['a'-'z' 'A'-'Z' '0'-'9' '_']* as var { VARIABLE(var) }
| ['A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']* as var { TABLELIT(var) }
| ['.']['a'-'z' 'A'-'Z' '0'-'9' '_']* as lxm { 
  let i = (String.length lxm) - 1 in 
  COLUMNLIT(String.sub lxm 1 i) 
}
| eof { EOF }

and comment = parse
  "*/" { tokenize lexbuf }
| _    { comment lexbuf }

and str buf = parse
  '"' { STRINGLIT(Buffer.contents buf) }
| [^ '"'] { Buffer.add_string buf (Lexing.lexeme lexbuf); str buf lexbuf }


