(* Semantically-checked Abstract Syntax Tree and functions for printing it *)

open Ast
type scell = 
 SIntLit of int
| SFloatLit of float
| SStrLit of string
| SBoolLit of bool

type stable = primitive * st
and st =
	SJoin of  string * stable * stable * string list * string list
	| SDistinct of stable * (primitive * string) list
	| SSelect of stable * (primitive * string) list
	| SInsert of stable * sexpr list
	| SCreate of (primitive * string) list
  | SReadFile of string * primitive list
  | STableLit of string
  | SWhere of stable * ((primitive * string) * operator * sexpr)
	| SDelete of stable * ((primitive * string) * operator * sexpr) 

and sexpr = primitive * sx
and sx =  
  SBinop of sexpr * operator * sexpr
| SNot of sexpr
| SVal of string
| SAssign of primitive * string * sexpr
| SReassign of string * sexpr
| SReturn of sexpr
| STableExpr of stable
| SCell of scell


type sstmt  =
  SConditionWithElse of sexpr * sstmt * sstmt
| SCondition of sexpr * sstmt
| SExpr of sexpr
| SWhile of sexpr * sstmt
| SDefineFunction of primitive * string * (primitive * string) list * sstmt
| SSemi of sstmt * sstmt
| SSemi1 of sstmt
| SPrint of sexpr


(* Pretty-printing functions *)
let string_of_scell = function
  SIntLit(l) -> string_of_int l
| SStrLit(l) -> l
| SBoolLit(true) -> "true"
| SBoolLit(false) -> "false"
| SFloatLit(l) -> string_of_float l

let string_of_decl (typ, name)= (string_of_typ typ) ^ " " ^ name

let string_of_insert_cell (typ, c)= (string_of_typ typ) ^ " " ^ (string_of_scell c)


let rec string_of_sexpr (t, e)= 
"(" ^ string_of_typ t ^ " : " ^ (match e with
    SBinop(e1, o, e2) -> string_of_sexpr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_sexpr e2
  | SNot(e) -> "Not" ^ string_of_sexpr e
  | SVal(e) -> e
  | SAssign(t, v, e) -> string_of_typ t ^ v ^ " = " ^ string_of_sexpr e
  | SReassign(v, e) -> v ^ " = " ^ string_of_sexpr e
  | SReturn(e) -> "return " ^ string_of_sexpr e ^ ";\n";
  | STableExpr(e) -> string_of_stable e
  | SCell(e) -> string_of_scell(e)
) ^ ")"	
and string_of_stable (t, e)= 
"(" ^ string_of_typ t ^ " : " ^ (match e with
	  SJoin(_,_,_,_,_) -> "Join"
	| SDistinct(_,_) -> "Distinct"
	| SSelect(l, c) -> string_of_stable l ^ ".SELECT(" ^ String.concat ", " (List.rev (List.map string_of_decl c)) ^")"
	| SInsert(t,c) -> string_of_stable t ^ ".INSERT(" ^ String.concat ", " (List.rev (List.map string_of_sexpr c)) ^ ")"
	| SCreate(a) -> "TABLE{\n\t" ^ String.concat ",\n\t" (List.rev (List.map string_of_decl a)) ^ "}"
  | STableLit(l) -> l
  | SWhere(l, (c, op, s)) -> string_of_stable l ^ ".WHERE(" ^ string_of_decl c ^ string_of_op op ^ (string_of_sexpr s) ^")"
	| SDelete(l, (c, op, s)) -> string_of_stable l ^ ".DELETE(" ^ string_of_decl c ^ string_of_op op ^(string_of_sexpr s) ^")"
  | SReadFile(a,c) -> "IMPORT \"" ^ a ^ "(" ^ String.concat ", " (List.rev (List.map string_of_typ c)) ^")\n"
) ^ ")"


let rec string_of_sstmt = function
  SConditionWithElse(e,s1,s2) -> "if (" ^ string_of_sexpr e ^ "){\n"^ string_of_sstmt s1 ^ "\n} else{\n" ^string_of_sstmt s2 ^ "\n}\n" 
| SCondition(e,s) -> "if (" ^ string_of_sexpr e ^ "){\n"^ string_of_sstmt s ^ "\n}\n" 
| SExpr(e) -> string_of_sexpr e
| SWhile(e,s) -> "while("^ string_of_sexpr e ^"){\n"^ string_of_sstmt s^"\n}\n"
| SDefineFunction(_,_,_,_) -> "function"
| SSemi1(e) -> string_of_sstmt e ^ ";\n"
| SSemi(e,e2) -> string_of_sstmt e ^ ";\n" ^ string_of_sstmt e2
| SPrint(e) -> "Print" ^ string_of_sexpr e

