type operator = Add | Sub | Mul | Div | Equ | Neq	| And	| Or | Lt	| Gt | Gteq | Lteq


type primitive =
  Void
| Int
| Float
| Bool
| String
| Table of (primitive * string) list
| Column

type cell = 
   IntLit of int
  | FloatLit of float
  | StrLit of string
  | BoolLit of bool

type table = 
	Join of  string * table * table * string list * string list
	| Distinct of table * string list
	| Select of table * string list
	| Where of table * (string * operator * expr) 
	| Delete of table * (string * operator * expr) 
	| Insert of table * expr list
	| Create of (primitive * string) list
  | ReadFile of string * primitive list
  | TableLit of string

and expr =  
  Binop of expr * operator * expr
  | Not of expr
  | Val of string
  | Assign of primitive * string * expr
  | Reassign of string * expr
  | Return of expr
  | TableExpr of table
  | Cell of cell

type stmt  =
  ConditionWithElse of expr * stmt * stmt
  | Condition of expr * stmt
  | Expr of expr
  | While of expr * stmt
  | DefineFunction of primitive * string * (primitive * string) list * stmt
  | Semi1 of stmt
  | Semi of stmt * stmt
  | Print of expr


  (* Pretty-printing functions *)


let rec string_of_typ = function
  Int -> "int"
| Bool -> "boolean"
| Float -> "float"
| Void -> "void"
| String -> "string"
| Table t -> "TABLE <"^ String.concat ", " (List.rev (List.map string_of_decl t)) ^ ">"
| Column -> "column"
and string_of_decl (typ, name)= (string_of_typ typ) ^ " " ^ name

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mul -> "*"
  | Div -> "/"
  | Equ -> "=="
  | Neq -> "!="
  | Lt -> "<"
  | Lteq -> "<="
  | Gt -> ">"
  | Gteq -> ">="
  | And -> "AND"
  | Or -> "||"

let string_of_cell = function
  IntLit(l) -> string_of_int l
| StrLit(l) -> l
| BoolLit(true) -> "true"
| BoolLit(false) -> "false"
| FloatLit(l) -> string_of_float l



let rec string_of_table =  function
	  Join(_,_,_,_,_) -> "Join"
	| Distinct(_,_) -> "Distinct"
	| Select(l, c) -> string_of_table l ^ ".SELECT(" ^ String.concat "," c ^")"
	| Where(_,_) -> "WHERE"
	| Delete(_,_) -> "Delete"
	| Insert(t,c) -> string_of_table t ^ ".INSERT(" ^ String.concat ", " (List.rev (List.map string_of_expr c)) ^ ")"
	| Create(d) -> "TABLE{\n\t" ^ String.concat ",\n\t" (List.rev (List.map string_of_decl d)) ^ "\n}\n"
  | ReadFile(a,c) -> "IMPORT \"" ^ a ^ "(" ^ String.concat ", " (List.rev (List.map string_of_typ c)) ^")\n"
  | TableLit(l) -> l
and string_of_expr = function
| Binop(e1, o, e2) ->
    string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
| Not(e) -> "Not " ^ string_of_expr e
| Assign(t, v, e) -> string_of_typ t ^ " " ^ v ^ " = " ^ string_of_expr e
| Reassign(v, e) -> v ^ " = " ^ string_of_expr e
| Val(e) -> e
| Return(e) -> "return " ^ string_of_expr e
| TableExpr(l) -> string_of_table l
| Cell(e) -> string_of_cell(e)



let rec string_of_stmt = function
  Condition(e, s) -> "if (" ^ string_of_expr e ^ ")\n{"^ string_of_stmt s ^ "\n}" 
| ConditionWithElse(e, s1, s2) ->  "if (" ^ string_of_expr e ^ "){\n"^ string_of_stmt s1 ^ "\n} else{" ^string_of_stmt s2 ^ "\n}\n" 
| Expr(e) -> string_of_expr e
| While(e,s) -> "while "^ string_of_expr e ^ string_of_stmt s
| DefineFunction(_,_,_,_) -> "function"
| Semi1(e) -> string_of_stmt e ^ ";\n"
| Semi(e, e2) -> string_of_stmt e ^ ";\n" ^ string_of_stmt e2
| Print(e) -> "Print" ^ string_of_expr e
  

let string_of_vdecl (t, id) = string_of_typ t ^ " " ^ id ^ ";\n"

let string_of_program (a) =
  string_of_stmt a
