module Lib where

import Control.Applicative ((<|>))
import Data.Set (Set)
import qualified Data.Set as Set
import Data.Maybe (mapMaybe, catMaybes)

data Expr = Var String
   | And Expr Expr
   | Or Expr Expr
   | Not Expr
   | Const Bool
  deriving (Show, Eq)

-- Return the first free variable in the expression.
freeVariable :: Expr -> Maybe String
freeVariable (Const _) = Nothing
freeVariable (Var v) = Just v
freeVariable (Not e) = freeVariable e
freeVariable (Or x y) = freeVariable x <|> freeVariable y
freeVariable (And x y) = freeVariable x <|> freeVariable y

guessVariable :: String -> Bool -> Expr -> Expr
guessVariable var val expr =
   case expr of
      Var v -> if v == var
              then Const val
              else Var v
      Not expr' -> Not (guess expr')
      Or x y -> Or (guess x) (guess y)
      And x y -> And (guess x) (guess y)
      Const b -> Const b
   where guess = guessVariable var val

-- Recursively evaluate the expression until we arrive at the Variable or a Boolean Value 
simplify :: Expr -> Expr
simplify (Const b) = Const b
simplify (Var v) = Var v
simplify (Not expr) =
  case simplify expr of
     Const b -> Const (not b)
     expr' -> Not expr'
simplify (Or x y) =
   let es = filter (/= Const False) [simplify x, simplify y] in
        if Const True `elem` es
        then Const True
        else
          case es of
            []       -> Const False
            [e]      -> e
            [e1, e2] -> Or e1 e2
            _        -> error "Should never happen."
simplify (And x y) =
   let es = filter (/= Const True) [simplify x, simplify y] in
        if Const False `elem` es
        then Const False
        else
          case es of
            []       -> Const True
            [e]      -> e
            [e1, e2] -> And e1 e2
            _        -> error "Should never happen."

-- Unwrap the Boolean from the DataType
unConst :: Expr -> Bool
unConst (Const b) = b
unConst _ = error "Not Const"

-- Remove Negations - apply De Morgan's Law
fixNegations :: Expr -> Expr
fixNegations expr =
   case expr of
      Not (Not x) -> fixNegations x
      Not (And x y) -> Or (fixNegations $ Not x) (fixNegations $ Not y)
      Not (Or x y) -> And (fixNegations $ Not x) (fixNegations $ Not y)
      Not (Const b) -> Const (not b)
      Not x -> Not (fixNegations x)
      And x y -> And (fixNegations x) (fixNegations y)
      Or x y -> Or (fixNegations x) (fixNegations y)
      x -> x

-- Unwrap the Literals in the expression
literals :: Expr -> Set String 
literals (Var v) = Set.singleton v
literals (Not e) = literals e
literals (And x y) = Set.union (literals x) (literals y)
literals (Or x y) = Set.union (literals x) (literals y)
literals _ = Set.empty

data Polarity = Positive | Negative | Mixed deriving (Show, Eq)

-- Find the polarities of the literals in the expression
literalPolarity :: Expr -> String -> Maybe Polarity
-- positive polarity
literalPolarity (Var v) v'
    | v == v' = Just Positive
    | otherwise = Nothing
-- negative polarity
literalPolarity (Not (Var v)) v'
    | v == v' = Just Negative
    | otherwise = Nothing
-- recursively find polarities in And and Or constructors
literalPolarity expr v =
   case expr of
      And x y -> combinePolarities [x, y]
      Or x y  -> combinePolarities [x, y]
      Not x   -> error $ "Not in CNF: negation of a non-literal: " ++ show x
      Const _ -> Nothing
      _       -> error "Should never happen."
   where
      combinePolarities es =
        let polarities = mapMaybe (flip literalPolarity v) es
        in case polarities of
          [] -> Nothing
          ps -> if all (== Positive) ps
                then Just Positive
                else if all (== Negative) ps
                     then Just Negative
                     else Just Mixed


literalElimination :: Expr -> Expr
literalElimination e =
    let ls = Set.toList (literals e)
        ps = map (literalPolarity e) ls

        -- Determine Polarity that needs to be assigned to the Literal 
        extractPolarized :: String -> Maybe Polarity -> Maybe (String, Bool)
        extractPolarized v (Just Positive) = Just (v, True)
        extractPolarized v (Just Negative) = Just (v, False)
        extractPolarized _ _ = Nothing

        -- Gives you all the Polarity Assignments of each Literal
        assignments :: [(String, Bool)]
        assignments = catMaybes $ zipWith extractPolarized ls ps

        -- Replace the literals with a Boolean Value
        replacers :: [Expr -> Expr]
        replacers = map (uncurry guessVariable) assignments
        replaceAll :: Expr -> Expr
        replaceAll = foldl (.) id replacers
    in replaceAll e

-- Find the clauses where there is only 1 literal in the clause
unitClause :: Expr -> Maybe (String, Bool)
unitClause (Var v) = Just (v, True)
unitClause (Not (Var v)) = Just (v, False)
unitClause _ = Nothing

-- Create a list of clauses by travesing the tree of And constructors
clauses :: Expr -> [Expr]
clauses (And x y) = clauses x ++ clauses y
clauses expr = [expr]

-- Extract all unit clauses 
allUnitClauses :: Expr -> [(String, Bool)]
allUnitClauses = mapMaybe unitClause . clauses

-- this will replace all unit clauses with the appropiate Boolean Value
unitPropagation :: Expr -> Expr
unitPropagation expr = replaceAll expr
    where
      assignments :: [(String, Bool)]
      assignments = allUnitClauses expr
      replaceAll :: Expr -> Expr
      replaceAll = foldl (.) id (map (uncurry guessVariable) assignments)