module Lib where

import Data.List.Split(splitOn)
import qualified Data.Set as Set
import Data.Set (Set)
import qualified Data.Map as Map
import Data.Map (Map)
import Data.List
import Control.Parallel.Strategies(NFData, rpar, withStrategy, parBuffer, rdeepseq, parList, using)

------------------------------------------------------------------------
-- Parsing
-- parses the Ints from the Chars
clean :: [Char] -> [Int]
clean row = map (\word -> read word::Int) $ splitOn "," row

-- reads in the content of the file, outputs True if puzzle is solved, False otherwise
nonogram :: String -> Bool
nonogram puzzle_board = 
    let info = init.tail $ dropWhile (/="") $ lines puzzle_board in
    let h = map (\line -> clean line ) $ tail $ takeWhile (/= "") info in
    let v = map (\line -> clean line ) $ tail $ filter (/= "") (dropWhile (/= "") info) in
    check $ solve (puzzle h v)

------------------------------------------------------------------------
-- Cells

newtype Value = Value Int
    deriving (Eq, Ord, Show)

-- | Negative values encode empty cells, positive values filled cells
empty :: Value -> Bool
empty (Value n) = n <= 0

full :: Value -> Bool
full = not . empty

type Choice = Set Value

------------------------------------------------------------------------
-- Puzzle

type Grid = [[Choice]]

-- | Datatype for solved and unsolved puzzles
data Puzzle = Puzzle
    -- | List of rows, containing horizontal choices for each cell
    { gridH :: Grid
    -- | List of columns, containing vertical choices for each cell
    , gridV :: Grid
    -- | What is allowed before/after a specific value?
    --   (after (Value 0)) are the values allowed on the first position
    , afterH, beforeH :: [Value -> Choice]
    , afterV, beforeV :: [Value -> Choice]
    }

instance Eq Puzzle where
    p == q = gridH p == gridH q

instance Show Puzzle where
    show = dispGrid . gridH

-- | Transpose a puzzle (swap horizontal and vertical components)
transposeP :: Puzzle -> Puzzle
transposeP p = Puzzle
    { gridH      = gridV p
    , gridV      = gridH p
    , afterH     = afterV p
    , beforeH    = beforeV p
    , afterV     = afterH p
    , beforeV    = beforeH p
    }
    
-- | Display a puzzle
dispGrid :: [[Set Value]] -> [Char]
dispGrid = concatMap (\r -> "[" ++ map disp'' r ++ "]\n")
 where disp'' x
        | Set.null     x = 'E'
        | setAll full  x = '1'
        | setAll empty x = '0'
        | otherwise      = '/'


------------------------------------------------------------------------
-- Making puzzles

-- | Generate puzzle
puzzle :: [[Int]] -> [[Int]] -> Puzzle
puzzle h v = Puzzle
    { gridH   = gH
    , gridV   = gV
    , afterH  = fst abH
    , beforeH = snd abH
    , afterV  = fst abV
    , beforeV = snd abV
    }
 where rows = length h
       cols = length v
       ordersH = map order h
       ordersV = map order v
       (abH, abV) = (beforeAfter ordersH, beforeAfter ordersV)
       (gH, gV) = (getGrid cols ordersH, getGrid rows ordersV)

getGrid :: Ord a => Int -> [[a]] -> [[Set a]]
getGrid numCells orders = map(replicate numCells . Set.fromList) orders

beforeAfter :: [[Value]] -> ([Value -> Choice], [Value -> Choice])
beforeAfter orders = (after, before)
    where before = map mkAfter $ map reverse orders
          after = map mkAfter orders

-- | Gets possible values for a line in order
order :: [Int] -> [Value]
order = order' 1
 where order' n []  = [Value (-n), Value (-n)]
       order' n (x:xs) = [Value (-n), Value (-n)] ++ map Value [n..n+x-1] ++ order' (n+x) xs

mkAfter :: [Value] -> Value -> Choice
mkAfter ord = (mkAfterM ord Map.!)

mkAfterM :: [Value] -> Map Value (Set Value)
mkAfterM ord  = Map.fromListWith (Set.union) aftersL
 where aftersL =
                (if length ord > 2
                then [(Value 0, Set.singleton (ord !! 2))]
                else []) ++
              zip (Value 0:ord) (map Set.singleton ord)

------------------------------------------------------------------------
-- Checking puzzles

check :: [Puzzle] -> Bool
check ps
    | length ps == 0    = False
    | invalid $ head ps = False
    | done $ head ps    = True
    | otherwise         = False

done :: Puzzle -> Bool
done = all (all ((==1) . Set.size)) . gridH

invalid :: Puzzle -> Bool
invalid = any (any Set.null) . gridH

------------------------------------------------------------------------
-- Algorithm Stepping

-- | Deterministic solving
solveD :: Puzzle -> Puzzle
solveD = consecSame . iterate step
 
-- | Combine steps
step :: Puzzle -> Puzzle
step = hvStep . transposeP . lineStep . transposeP . lineStep

-- | Single step
lineStep :: Puzzle -> Puzzle
lineStep p = p { gridH = gridH'' }
 where gridH'  = zipWith lineStepFwd (afterH  p) (gridH p)
       gridH'' = zipWith lineStepBack (beforeH p) (gridH')

-- | lineStep on a single line forward and backward
lineStepFwd :: (Value -> Set Value) -> [Set Value] -> [Set Value]
lineStepFwd after row = lineStepFwd' (after (Value 0)) row
 where lineStepFwd' _         []     = []
       lineStepFwd' afterPrev (x:xs) = x' : lineStepFwd' afterX' xs
        where x' = Set.intersection x afterPrev
              afterX' = Set.unions $ withStrategy (parBuffer 100 rpar) $ map after $ Set.toList x'

lineStepBack :: (Value -> Set Value) -> [Set Value] -> [Set Value]
lineStepBack before = reverse . lineStepFwd before . reverse

-- | Sharing information between the horizontal grid and vertical grid
hvStep :: Puzzle -> Puzzle
hvStep p = p { gridH = gridH', gridV = transpose gridV't }
 where (gridH', gridV't) = zMap (zMap singleStep) (gridH p) (transpose (gridV p))

-- Step on a single cell
singleStep :: Set Value -> Set Value -> (Set Value, Set Value)
singleStep h v = filterCell empty . filterCell full $ (h,v)

-- Step on a single cell, for a single condition, if either h or v satisfies the condition
-- then the other is filtered so it will satisfy as well
filterCell :: (a -> Bool) -> (Set a, Set a) -> (Set a, Set a)
filterCell cond (h,v) 
    | setAll cond h = (h, Set.filter cond v)
    | setAll cond v = (Set.filter cond h, v)
    | otherwise     = (h, v)

------------------------------------------------------------------------
-- Nondeterministic

-- | Solve a puzzle, gives all solutions
solve :: Puzzle -> [Puzzle]
solve p
 | all (all ((==1) . Set.size)) . gridH $ p' = [p'] -- single solution
 | invalid p' = []        -- no solutions
 | otherwise       = concatMap solve (guess p') -- we have to guess
 where p' = solveD p

-- | Branch out to multiple possible choices for grids

guess :: Puzzle -> [Puzzle]
guess p = map (\gh -> p {gridH = gh} ) gridHs
 where gridHs = getMultiple (getMultiple getChoices) (gridH p)

-- | Gets multiple possible choices for a single cell
getChoices :: Choice -> [Choice]
getChoices = map Set.singleton . Set.toList

-- | Tries to split a single item in a list using the function f
--   Stops at the first position where f has more than 1 result.
getMultiple :: (a -> [a]) -> [a] -> [[a]]
getMultiple _ []     = []
getMultiple f (x:xs)
 | length fx  > 1 = map (:xs) fx
 | length fxs > 1 = map (x:) fxs
 | otherwise      = []
 where fx  = f x
       fxs = getMultiple f xs

------------------------------------------------------------------------
-- Utilities

-- | parallelization, especially on zMap
par' :: NFData a => [a] -> [a]
par' = (`using` parList rdeepseq)

-- Examples of some other strategies that we tried
-- parPair2 = do 
-- 	 evalTuple2 (rparWith rdeepseq) (rparWith rdeepseq)

-- parRds :: NFData a => [a] -> [a]
-- parRds = (`using` parBuffer 250 rdeepseq)

-- parPair :: Strategy (a,b)
-- parPair (a,b) = do
--    a' <- rpar a
--    b' <- rpar b
--    return (a',b')

-- | Set.all, similar to Data.List.all
setAll :: (a -> Bool) -> Set a -> Bool
setAll f = all f . Set.toList

-- | A zip-like map
zMap :: (a -> b -> (c, d)) -> [a] -> [b] -> ([c], [d])
zMap f a b = unzip $ zipWith f a b

-- | Find the first item in a list that is repeated
consecSame :: Eq a => [a] -> a
consecSame (a:b:xs)
 | a == b    = a
 | otherwise = consecSame (b:xs)

consecSame _ = error "Invalid"
