{-
    nonogram_solver.hs
    @author Manav Goel (mg3851) and Tanvi Hisaria (th2720)
-}

import System.Environment(getArgs, getProgName)
import Control.Monad (when, mplus, foldM)
import Control.Parallel (par, pseq)
import Data.IntMap (IntMap, insert, toList, fromList, (!))

data Color = White | Black
           deriving (Eq)
instance Show Color where
    show Black = "X"
    show White = "-"

type Nonogram = [[Color]]
type Hint = [Int]   -- represents a hint for a row or column e.g. [2, 4, 5]
data ColumnInfo = PlacedColor Color -- A single filled cell
                | BlackRun Int      -- Length of next black cell run
type ColumnInfoMap = IntMap [ColumnInfo]

-- Checks if a Column has any Black squares remaining
isColumnEmpty :: [ColumnInfo] -> Bool
isColumnEmpty [] = True
isColumnEmpty (PlacedColor Black : _) = False
isColumnEmpty (BlackRun _ : _) = False
isColumnEmpty (_ : xs) = isColumnEmpty xs

-- Tries consuming a color in a column, returns the correct column if valid
tryPlacingColor :: Color -> [ColumnInfo] -> Maybe [ColumnInfo]
tryPlacingColor White [] = Just []
tryPlacingColor Black [] = Nothing
-- Consume a matching PlacedColor
tryPlacingColor y (PlacedColor x : hs) = if x == y then Just hs else Nothing
tryPlacingColor White hs = Just hs
-- Expand a BlackRun if we start one
tryPlacingColor Black (BlackRun n : hs) = Just $ replicate (n - 1) (PlacedColor Black) ++ (PlacedColor White : hs)

-- Just grabs the correct column from the map to pass to tryPlacingColor
placeColorHelper :: Color -> ColumnInfoMap -> Int -> Maybe ColumnInfoMap
placeColorHelper colorToTry columnMap index = do
    hs <- tryPlacingColor colorToTry $ columnMap ! index
    return $ insert index hs columnMap

-- Main recursive driver for solving a puzzle
solve :: Int -> Int -> [Hint] -> ColumnInfoMap -> Maybe Nonogram
solve width columnIndex rowHints columnMap
        | null rowHints =
            if all isColumnEmpty (map snd $ toList columnMap)
                then return [[]]    -- All hints and ColumnInfo exhausted, puzzle is solved
                else Nothing        -- If there are still unplaced squares in the ColumnInfoMap, this puzzle is invalid
        | null hint = do -- This specific hint has been exhausted, meaning the row is complete
            updatedInfoMap <- foldM (placeColorHelper White) columnMap [columnIndex .. width - 1]
            rows <- solve width 0 remainingHints updatedInfoMap    -- start solving the next row
            return $ replicate (width - columnIndex) White : rows
        | otherwise -- Try to place a black and white square next in parallel, keep the one that works
            = tryPlaceBlack `par` tryPlaceWhite `pseq` mplus tryPlaceBlack tryPlaceWhite
            
        where
            (hint : remainingHints) = rowHints
            (h : hs) = hint

            tryPlaceBlack = do
                -- The current hint extends past the end of the puzzle
                when (columnIndex + h > width) Nothing

                -- Try to add h black cells to the row
                updatedInfoMap <- foldM (placeColorHelper Black) columnMap [columnIndex .. columnIndex + h - 1]

                -- Try to place a white cell if we aren't at the end of the row
                im' <- if columnIndex + h == width
                        then return updatedInfoMap
                        else placeColorHelper White updatedInfoMap (columnIndex + h)

                -- Solve the rest of the row
                (row : rows) <- solve width (columnIndex + h + 1) (hs : remainingHints) im'
                let row' = if columnIndex + h == width 
                            then row 
                            else White : row

                return $ (replicate h Black ++ row') : rows

            tryPlaceWhite = do
                when (columnIndex >= width) Nothing
                updatedInfoMap <- placeColorHelper White columnMap columnIndex
                
                -- Solve the rest of the row
                (row : rows) <- solve width (columnIndex + 1) rowHints updatedInfoMap
                return $ ((White : row)) : rows

-- Helper that creates the ColumnInfoMap and passes it to solve
nonogram :: [[Hint]] -> Maybe Nonogram
nonogram [] = Nothing
nonogram [_] = Nothing
nonogram (_:_:_:_) = Nothing
nonogram [rows, columns] = solve (length columns) 0 rows myColumnMap
  where
    myColumnMap = fromList (zip [0 ..] $ map (map BlackRun) columns)

printNonogram :: Maybe Nonogram -> IO ()
printNonogram Nothing  = putStrLn "No solution!"
printNonogram (Just s) = mapM_ (putStrLn . concatMap show) s

strToInt :: String -> [[[Int]]]
strToInt a = read a::[[[Int]]]

main :: IO ()
main = do
    args <- getArgs
    case args of
        [filename] -> do
            contents <- readFile filename
            let rawPuzzle = strToInt (lines contents !! 0)
            printNonogram $ nonogram rawPuzzle
        _ -> do
            name <- getProgName
            putStrLn $ "Usage: " ++ name ++ "<filename>"
