module Lib
    ( mandelbrotAsListS, mandelbrotAsListP, mandelbrotAsVectorS, 
      mandelbrotAsVectorP, mandelbrotAsRepaS, mandelbrotAsRepaP) 
    where

import Data.Complex ( magnitude, Complex((:+)) )
import Data.Vector ( generate, Vector )
import Data.Array.Repa ( fromFunction, computeS, computeP, Z (..), (:.)(..), Array, U, DIM2 )
import Data.Functor.Identity ( runIdentity )
import Control.Parallel.Strategies ( parListChunk, rdeepseq, using)
import Data.Vector.Strategies ( parVector )
import Data.Word ( Word8 )

mandelbrot :: Complex Double -> Word8
mandelbrot c = escapeTime
    where (_, escapeTime) = last $
                takeWhile (\(z, count) -> magnitude z < 2 && count < 100) $
                iterate   (\(z, count) -> (z * z + c, count + 1)) (0.0 :+ 0.0, 0)

pixelValue :: (Int, Int) -> (Int, Int) -> Complex Double
pixelValue (rowCount, colCount) (row, col)  = 
    shiftAlongX col colCount :+ shiftAlongY row rowCount 
    where shiftAlongX x maxX  = normalizeZeroOne x maxX 3 2 -- i.e. (-2, 1)
          shiftAlongY y maxY  = normalizeZeroOne y maxY 2 1 -- i.e. (-1, 1)
          normalizeZeroOne v m a b = a * (fromIntegral v - 1)/(fromIntegral m - 1) - b

toRgb :: Word8 -> (Word8, Word8, Word8)
toRgb i = if i == 0 then (0, 0, 0) else (r, g, b) 
    where i' = (floor (255 * 255 * (255.0::Double)/100.0)::Integer) * toInteger i
          r' = i' `mod` 255
          r = fromIntegral r'
          g = floor(fromIntegral (i' - r') / 255::Double) `mod` 255
          b = floor(fromIntegral (i' - r') / 
                (255 * 255::Double)) - floor(fromIntegral g / 255::Double)        



pixelToRGB :: (Int, Int) -> (Int, Int) -> (Word8, Word8, Word8)
pixelToRGB (row, col) = toRgb . mandelbrot . pixelValue (row, col)

---Sequential mandelbrot

mandelbrotAsListS :: Int -> Int -> [(Word8, Word8, Word8)]
mandelbrotAsListS rowCount colCount = [ pixelToRGB (rowCount, colCount) (r, c) 
                 | r <- [1..rowCount],  c <- [1..colCount]]
        

mandelbrotAsVectorS :: Int -> Int -> Vector (Word8, Word8, Word8)
mandelbrotAsVectorS rowCount colCount = Data.Vector.generate (rowCount * colCount) 
        (\n -> pixelToRGB (rowCount, colCount) (quotRem n colCount) )


mandelbrotAsRepaS :: Int -> Int -> Array U DIM2 (Word8, Word8, Word8)
mandelbrotAsRepaS rowCount colCount = computeS $ 
        fromFunction (Z :. (rowCount::Int) :. (colCount::Int)) 
                (\(Z :. row :. col) -> pixelToRGB (rowCount, colCount) (row, col) )


---Parallel mandelbrot

mandelbrotAsListP :: Int -> Int -> [(Word8, Word8, Word8)]
mandelbrotAsListP rowCount colCount = [ pixelToRGB (rowCount, colCount)(r, c)  
        | r <- [1..rowCount],  c <- [1..colCount]] 
                `using` parListChunk 1000 rdeepseq


mandelbrotAsVectorP :: Int -> Int -> Vector (Word8, Word8, Word8)
mandelbrotAsVectorP rowCount colCount = Data.Vector.generate (rowCount * colCount) 
        (\n -> pixelToRGB (rowCount, colCount) (quotRem n colCount) ) 
                `using` parVector 1000


mandelbrotAsRepaP :: Int -> Int -> Array U DIM2 (Word8, Word8, Word8)
mandelbrotAsRepaP rowCount colCount = runIdentity $ computeP $ 
        fromFunction (Z :. (rowCount::Int) :. (colCount::Int)) 
                (\(Z :. row :. col) -> pixelToRGB (rowCount, colCount) (row, col) )

