module Main where

import System.Environment ( getArgs )
import System.Exit ( die )
import Data.Word ( Word8 )
import Data.ByteString (pack ) 
import Data.Array.Repa ( extent)
import Data.Array.Repa.IO.BMP ( writeImageToBMP )
import Data.Time ( getCurrentTime, diffUTCTime )
import Control.DeepSeq ( force )
import Codec.BMP ( writeBMP, packRGBA32ToBMP, BMP )    

import Lib   ( mandelbrotAsListS, mandelbrotAsListP, mandelbrotAsVectorS, 
               mandelbrotAsVectorP, mandelbrotAsRepaS, mandelbrotAsRepaP ) 


main :: IO ()
main = do
    let errorMessage =  "Usage: final <choice> <row count> <column count>"
    args <- getArgs
    case args of
        [ch, rc, cc] ->  do
            let (choice, rowCount,colCount) = (read ch :: Int, read  rc :: Int, read  cc :: Int)

            beforeT <- getCurrentTime 
            case choice of
                c
                    | c >=  1 && c <=  6 -> mandelbrotSaveImage  choice rowCount colCount 
                    | c >=  7 && c <=  9 -> mandelbrotSequential choice rowCount colCount 
                    | c >= 10 && c <= 12 -> mandelbrotParallel   choice rowCount colCount 

                    | otherwise -> die errorMessage

            afterT <- getCurrentTime
            print $ show choice ++ " (" ++ show (diffUTCTime afterT beforeT) ++ ")"                    

        _ -> die errorMessage



mandelbrotSaveImage :: Int -> Int -> Int-> IO ()
mandelbrotSaveImage choice rowCount colCount = do

     case choice of
        c
            | c ==  1 -> writeBMP        "mandelbrotAsListS.bmp"   $ collectionToBmp     rowCount colCount 
                                                                   $ mandelbrotAsListS   rowCount colCount
            | c ==  2 -> writeBMP        "mandelbrotAsVectorS.bmp" $ collectionToBmp     rowCount colCount 
                                                                   $ mandelbrotAsVectorS rowCount colCount
            | c ==  3 -> writeImageToBMP "mandelbrotAsRepaS.bmp"   $ mandelbrotAsRepaS   rowCount colCount

            | c ==  4 -> writeBMP        "mandelbrotAsListP.bmp"   $ collectionToBmp     rowCount colCount 
                                                                   $ mandelbrotAsListP   rowCount colCount
            | c ==  5 -> writeBMP        "mandelbrotAsVectorP.bmp" $ collectionToBmp     rowCount colCount 
                                                                   $ mandelbrotAsVectorP rowCount colCount
            | c ==  6 -> writeImageToBMP "mandelbrotAsRepaP.bmp"   $ mandelbrotAsRepaP   rowCount colCount

            | otherwise -> print "unexpected"


mandelbrotSequential :: Int -> Int -> Int-> IO ()
mandelbrotSequential choice rowCount colCount = do
 
    case choice of
        c
            | c == 7    -> print $ Prelude.length $ force $ mandelbrotAsListS   rowCount colCount 
            | c == 8    -> print $ Prelude.length $ force $ mandelbrotAsVectorS rowCount colCount 
            | c == 9    -> print $ extent $                 mandelbrotAsRepaS   rowCount colCount

            | otherwise -> print "unexpected"
    return ()    


mandelbrotParallel :: Int -> Int -> Int-> IO ()
mandelbrotParallel choice rowCount colCount = do

    case choice of
        c
            | c == 10   -> print $ Prelude.length $ mandelbrotAsListP   rowCount colCount 
            | c == 11   -> print $ Prelude.length $ mandelbrotAsVectorP rowCount colCount 
            | c == 12   -> print $ extent         $ mandelbrotAsRepaP   rowCount colCount

            | otherwise ->  print "unexpected"
    return ()   



-- conversion functions 

collectionToBmp :: Foldable t => Int -> Int -> t (Word8, Word8, Word8) -> BMP
collectionToBmp rowCount colCount xs = packRGBA32ToBMP colCount rowCount $ 
        Data.ByteString.pack ( Prelude.concatMap (\(r, g, b) -> [r, g, b, 255])  xs)
