/* * Device driver for the ultranes system
 *
 * A Platform device implemented using the misc subsystem
 *
 * Zach Schuermann
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod ultranes.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree ultranes.c
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include "ultranes.h"

#define DRIVER_NAME "ultranes"

/*
 * Information about our device
 */
struct ultranes_dev {
	struct resource res; /* Resource: our registers */
	void __iomem *virtbase; /* Where registers can be accessed in memory */
    int addr;
    int v;
} dev;

/*
 * Handle ioctl() calls from userspace:
 */
static long ultranes_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	ultranes_arg_t vla;
    int low; 
    int high; 

	switch (cmd) {

	case ULTRANES_READ_ADDR:
        low = ioread8(dev.virtbase);
        high = ioread8(dev.virtbase + 1);
        // todo check return vals
	  	vla.addr = low | (high << 8);
		if (copy_to_user((ultranes_arg_t *) arg, &vla,
				 sizeof(ultranes_arg_t)))
			return -EACCES;
		break;

    case ULTRANES_SEND:
		if (copy_from_user(&vla, (ultranes_arg_t *) arg,
				 sizeof(ultranes_arg_t)))
			return -EACCES;
        iowrite8(vla.v, dev.virtbase + vla.addr);
        break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* The operations our device knows how to do */
static const struct file_operations ultranes_fops = {
	.owner		    = THIS_MODULE,
	.unlocked_ioctl = ultranes_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice ultranes_misc_device = {
	.minor		= MISC_DYNAMIC_MINOR,
	.name		= DRIVER_NAME,
	.fops		= &ultranes_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init ultranes_probe(struct platform_device *pdev)
{
	int ret;

	/* Register ourselves as a misc device: creates /dev/ultranes */
	ret = misc_register(&ultranes_misc_device);

	/* Get the address of our registers from the device tree */
	ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
	if (ret) {
		ret = -ENOENT;
		goto out_deregister;
	}

	/* Make sure we can use these registers */
	if (request_mem_region(dev.res.start, resource_size(&dev.res),
			       DRIVER_NAME) == NULL) {
		ret = -EBUSY;
		goto out_deregister;
	}

	/* Arrange access to our registers */
	dev.virtbase = of_iomap(pdev->dev.of_node, 0);
	if (dev.virtbase == NULL) {
		ret = -ENOMEM;
		goto out_release_mem_region;
	}
        
	return 0;

out_release_mem_region:
	release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
	misc_deregister(&ultranes_misc_device);
	return ret;
}

/* Clean-up code: release resources */
static int ultranes_remove(struct platform_device *pdev)
{
	iounmap(dev.virtbase);
	release_mem_region(dev.res.start, resource_size(&dev.res));
	misc_deregister(&ultranes_misc_device);
	return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id ultranes_of_match[] = {
	{ .compatible = "csee4840,ultranes-1.0" },
	{},
};
MODULE_DEVICE_TABLE(of, ultranes_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver ultranes_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(ultranes_of_match),
	},
	.remove	= __exit_p(ultranes_remove),
};

/* Called when the module is loaded: set things up */
static int __init ultranes_init(void)
{
	pr_info(DRIVER_NAME ": init\n");
	return platform_driver_probe(&ultranes_driver, ultranes_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit ultranes_exit(void)
{
	platform_driver_unregister(&ultranes_driver);
	pr_info(DRIVER_NAME ": exit\n");
}

module_init(ultranes_init);
module_exit(ultranes_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Zach Schuermann, Columbia University");
MODULE_DESCRIPTION("ultraNES driver");
