/*
 * VRAM fetcher
 * 
 * Currently ignoring sprite rendering
 * 
 * Inputs 
 * clk, reset, hcount, vcount, data from vram
 * vram_fetch_state 
 *   0 - NT byte
 *   1 - AT byte
 *   2 - low BG byte
 *   3 - high BG byte
 * 
 * Outputs
 * background data out
 *   BG_NT
 *   BG_AT
 *   BG_PT_1
 *   BG_PT_2
 */
module background_render( input logic         clk50,  // 50 MHz global clock
			  input logic 	      reset, // reset from cpu
			  input logic 	      ppu_en, //ppu_en for clock signal
			  input logic [8:0]   hcount, vcount, //from ppu_counters
			  input logic [7:0]   reg_x_scroll, reg_y_scroll, //$2005 read 2x
			  input logic [1:0]   nametable_select, //$2000 bits 0,1
			  input logic 	      pt_select, // $2000 bit 4
			  input logic [7:0]   vram_data_in, //data in from VRAM
			  input logic 	      vram_fetch, //is high when ppu access vram
			  input logic [2:0]   fetch_state, //what fetch step on
			  input logic 	      re_scroll, // flag from state table

			  output logic [3:0]  pixel_out, //4 bit color to MUX
			  output logic [13:0] address_out, // address to VRAM
			  output logic 	      rd // rd for vram 
			  );
   // internal data
   logic [7:0]	      nt_data; //data from the nametable
   logic [7:0] 	      at_data; //data from the attribute table
   logic [7:0] 	      pt_1_data; //lower half pattern table data
   logic [7:0] 	      pt_2_data; //upper half pattern tbale data
   //logic [7:0]        pattern_data_1; // 1st set of pattern data
   //logic [7:0]        pattern_data_2; // 2nd set of pattern data
   logic [7:0]        color_data_1; // 1st set of color data                
   logic [7:0] 	      color_data_2; // 2nd set of color data
   logic [15:0]       pattern_data_1_sr; // SR with 1st set of pattern data
   logic [15:0]       pattern_data_2_sr; // SR with 2nd set of pattern data
   logic [15:0]       color_data_1_sr; // SR with 1st set of color data
   logic [15:0]       color_data_2_sr; // SR with 2nd set of color data
   logic [13:0]       nt_address; // address in VRAM for nametable
   logic [13:0]       at_address; // address in VRAM for attribute table
   logic [13:0]       pt_address_1; // address for pattern table entry 1
   logic [13:0]       pt_address_2; // address for pattern table entry 2
   logic [3:0] 	      nt_select; // top 4 bits for nt_address
   logic [7:0] 	      at_select; // top 8 bits for at_address
   logic [2:0] 	      counter; // counter for data processing state machine
   logic [1:0] 	      at_bits; // for selecting which at data to use
   logic [15:0]       scroll;
   logic [7:0] 	      x_scroll; // x_sroll holder for incrementing
   logic [7:0] 	      y_scroll; // y_scroll holder for incrementing
   

   
   //                     //
   // address assignments //
   //                     //
   
   // name table & attribute table addresses
   always_ff @(posedge clk50 or posedge reset) begin
     if (reset) begin
	nt_select <= 4'd0;
	at_select <= 8'd0;
     end else begin
	case (nametable_select) 
	  2'b00 : begin
	     nt_select <= 4'b1000;
	     at_select <= 8'b10001111;
	  end 2'b01 : begin
	     nt_select <= 4'b1001;
	     at_select <= 8'b10011111;
	  end 2'b10 : begin
	     nt_select <= 4'b1010;
	     at_select <= 8'b10101111;
	  end 2'b11 : begin
	     nt_select <= 4'b1011;
	     at_select <= 8'b10111111;
	  end
	endcase // case (nametable_select)
     end // else: !if(reset)
   end // always_ff @

   assign nt_address [13:10] = nt_select;
   assign nt_address [9:5]   = y_scroll [7:3];
   assign nt_address [4:0]   = x_scroll [7:3];

   assign at_address [13:6]  = at_select;
   assign at_address [5:3]   = y_scroll [7:5];
   assign at_address [2:0]   = x_scroll [7:5];

   // pt address assignments
   assign pt_address_1 [13]   = 1'b0;
   assign pt_address_1 [12]   = pt_select;
   assign pt_address_1 [11:4] = nt_data;
   assign pt_address_1 [3]    = 1'b0;
   assign pt_address_1 [2:0]  = y_scroll [2:0];

   assign pt_address_2 [13]   = 1'b0;
   assign pt_address_2 [12]   = pt_select;
   assign pt_address_2 [11:4] = nt_data;
   assign pt_address_2 [3]    = 1'b1;
   assign pt_address_2 [2:0]  = y_scroll [2:0];


   //                      //
   // scroll incrementing  //
   //                      //
   assign x_scroll = scroll[7:0];
   assign y_scroll = scroll[15:8];
   
   always_ff @(posedge clk50 or posedge reset) begin
      if (reset) begin //|| ((hcount == 0) && (vcount == 0))) begin
	 scroll[7:0] <= reg_x_scroll;
	 scroll[15:8] <= reg_y_scroll;
      /*end else if (hcount == 0) begin //needs to become a flag
	 if (vcount == 0) begin
	    x_scroll <= reg_x_scroll;
	    y_scroll <= reg_y_scroll;
      end */
      //end else if (re_scroll) begin
	 //x_scroll <= reg_x_scroll;
	 // y_scroll <= reg_y_scroll;
      end else if (ppu_en) begin // & vram_fetch) begin
	     //{y_scroll, x_scroll} <= {y_scroll, x_scroll} + 1'b1;
	 scroll <= scroll + 1'b1;
	 //y_scroll <= y_scroll + reg_y_scroll + vcount;
      end
   end
   

   
   //                      //
   // Memory fetching      //
   //                      //

   always_ff @(posedge clk50 or posedge reset) begin
      if (reset) begin
	 // sero out all data
	 address_out <= 14'd0;
	 nt_data     <= 8'd0;
	 at_data     <= 8'd0;
	 pt_1_data   <= 8'd0;
	 pt_2_data   <= 8'd0;
      end else if (ppu_en & vram_fetch) begin
	 rd = 1'b1;
	 case (fetch_state[2:1])
	   2'b00 : begin
	      address_out <= nt_address;
	      nt_data     <= vram_data_in;
	   end
	   2'b01 : begin
	      address_out <= at_address;
	      at_data     <= vram_data_in;
	   end
	   2'b10 : begin
	      address_out <= pt_address_1;
	      pt_1_data   <= vram_data_in;
	   end
	   2'b11 : begin
	      address_out <= pt_address_2;
	      pt_2_data   <= vram_data_in;
	   end
	 endcase // case (fetch_state)
      end else// if (ppu_en and vram_fetch)
	rd <= 1'b0;
   end // always_ff @



   
   //                    //
   // Data processing    //
   //                    //

   // counter for data processing
   always_ff @(posedge clk50 or posedge reset)
      if (reset) counter <= 3'd0;
      else if (fetch_state == 3'd0) counter <= 3'd0; //sync with vram state machine
      else if (ppu_en) counter = counter + 1'b1;

   // Need to set which attribute bits we want to use
   // because of how the tiles are set up need to look at second LSB of xscroll
   // and y scroll. We can use those to determine which square we are in and set
   // tiles accordingly
   assign at_bits [0] = nt_address[1];
   assign at_bits [1] = nt_address[6];
   
   
   always_ff @(posedge clk50 or posedge reset) begin
     if (reset) begin
	//zero out all SRs
	color_data_1 <= 8'd0;
	color_data_2 <= 8'd0;
	color_data_1_sr <= 16'd0;
	color_data_2_sr <= 16'd0;
	pattern_data_1_sr <= 16'd0;
	pattern_data_2_sr <= 16'd0;
     end else if (ppu_en) begin //need to and and statmenet for background rendering
	// shift SRs every time
	color_data_1_sr          <= color_data_1_sr >> 1;
	color_data_2_sr          <= color_data_2_sr >> 1;
	pattern_data_1_sr        <= pattern_data_1_sr >> 1;
	pattern_data_2_sr        <= pattern_data_2_sr >> 1;
	
        case (counter)
          4'd0 : begin	       
	     // load SRs with data aquired last cycle
	     color_data_1_sr [15:8]   <= color_data_1;
	     color_data_2_sr [15:8]   <= color_data_2;
	     pattern_data_1_sr [15:8] <= pt_1_data;
	     pattern_data_2_sr [15:8] <= pt_2_data;
	  end
	  4'd4 : begin
	     // fill color data based on at_bits
	     case (at_bits)
	       2'b00 : begin
		  color_data_1 <= { at_data[0], at_data[0], at_data[0], at_data[0],
                                    at_data[0], at_data[0], at_data[0], at_data[0] };
		  color_data_2 <= { at_data[1], at_data[1], at_data[1], at_data[1], 
                                    at_data[1], at_data[1], at_data[1], at_data[1] };
	       end
	       2'b01 : begin
		  color_data_1 <= { at_data[2], at_data[2], at_data[2], at_data[2],
                                    at_data[2], at_data[2], at_data[2], at_data[2] };
		  color_data_2 <= { at_data[3], at_data[3], at_data[3], at_data[3],
                                    at_data[3], at_data[3], at_data[3], at_data[3] };
	       end
	       2'b10 : begin
		  color_data_1 <= { at_data[4], at_data[4], at_data[4], at_data[4],
                                    at_data[4], at_data[4], at_data[4], at_data[4] };
		  color_data_2 <= { at_data[5], at_data[5], at_data[5], at_data[5],
                                    at_data[5], at_data[5], at_data[5], at_data[5] };
	       end
	       2'b11 : begin
		  color_data_1 <= { at_data[6], at_data[6], at_data[6], at_data[6],
                                    at_data[6], at_data[6], at_data[6], at_data[6] };
		  color_data_2 <= { at_data[7], at_data[7], at_data[7], at_data[7],
                                    at_data[7], at_data[7], at_data[7], at_data[7] };
	       end
	     endcase // case (at_bits)
	  end // case: 4'd4
	endcase // case (counter)
     end // if (ppu_en)
   end // always_ff @

   // assign output
   assign pixel_out [0] = pattern_data_1_sr [0];
   assign pixel_out [1] = pattern_data_2_sr [0];
   assign pixel_out [2] = color_data_1_sr [0];
   assign pixel_out [3] = color_data_2_sr [0];
	  
endmodule
	   
