module Regex (runRegex) where

import Data.Char(isDigit, isAlphaNum)
import Data.List(isInfixOf, isPrefixOf)


runRegex :: String -> [String] -> [String]
runRegex regex contents =
    let res = map (matchRegex regex) contents
    in map (\(b, word) -> word) (filter (\(b, word) -> b) (zip res contents))

-- MATCH REGEX
-- basics
matchRegex :: String -> String -> Bool
matchRegex ('.':tlReg) (_:tlStr)     = matchRegex tlReg tlStr

-- or
matchRegex ('(':a:'|':b:')':tl) (hdStr:tlStr)
  | hdStr == a || hdStr == b         = matchRegex tl tlStr
  | otherwise                        = False
matchRegex ('(':_:'|':_:')':_) []    = False

-- kleene star
matchRegex regex@(a:'*':tl) (hdStr:tlStr)
  | a /= hdStr && tlStr == []        = False
  | a /= hdStr                       = matchRegex tl tlStr
  | a == hdStr                       = matchRegex tl tlStr || matchRegex regex tlStr

-- parenthesized kleene star
matchRegex regex@('(':_:_) str       = recurringKleene (regex) str []

-- escape characters
matchRegex ('\\':'d':tl) (hdStr:tlStr)
  | isDigit hdStr                    = matchRegex tl tlStr
  | otherwise                        = False

-- alphanum
matchRegex regex@(hdReg:tlReg) str@(hdStr:tlStr)
  | isAlphaNum' regex                = isInfixOf regex str
  | isAlphaNum hdReg && isAlphaNum hdStr && hdReg == hdStr
                                     = matchRegex tlReg tlStr
  | otherwise                        = False

matchRegex [] []                     = True
matchRegex [] _                      = False
matchRegex _ []                      = False
-- MATCH REGEX

-- HELPER FUNCTIONS
isAlphaNum' :: String -> Bool
isAlphaNum' (hd:tl)
  | isAlphaNum hd = isAlphaNum' tl
  | otherwise     = False
isAlphaNum' []    = True

removeKleene :: String -> String -> String
removeKleene kleene str = reverse (take ((length str) - (length $ tail kleene)) (reverse str))

recurringKleene :: String -> String -> String -> Bool
recurringKleene (')':'*':tl) str kleene
  | isPrefixOf (tail kleene) str
                       = (matchRegex (kleene ++ ")*" ++ tl) (removeKleene kleene str) || matchRegex tl str)
  | otherwise          = matchRegex tl str
recurringKleene (hd:tl) str kleene
                       = recurringKleene tl str (kleene ++ [hd])
recurringKleene [] _ _ = error "TODO how would this get called?"
-- HELPER FUNCTIONS
