{-# LANGUAGE DeriveGeneric #-}

import System.Environment(getArgs, getProgName)
import Data.Map(fromList, (!))
import Data.Maybe (catMaybes)
import System.Exit(die)
import Data.Char(isSpace)
import Data.List(minimumBy, minimum, maximum, partition, isInfixOf)
import Control.Parallel.Strategies(using, parList, rseq, rpar, runEval, rdeepseq)
import Control.DeepSeq
import GHC.Generics (Generic)

{-
Section I
Basic data types and operations used in the program
-}

deg2Rad :: Float
deg2Rad = 0.0174533

data Vector3 = Vector3 Float Float Float deriving (Eq, Generic)
instance Show Vector3 where
  show (Vector3 x y z) = "(" ++ show x ++ " , " ++ show y ++ " , " ++ show z ++ ")\n"

instance Num Vector3 where
  Vector3 x1 y1 z1 + Vector3 x2 y2 z2 = Vector3 (x1+x2) (y1+y2) (z1+z2)
  Vector3 x1 y1 z1 * Vector3 x2 y2 z2 = Vector3 (x1*x2) (y1*y2) (z1*z2)
  negate (Vector3 x y z) = Vector3 (-x) (-y) (-z)
  abs = error "not implemented"
  signum = error "not implemented"
  fromInteger = error "not implemented"

instance NFData Vector3

uniformVector :: Float -> Vector3
uniformVector a = Vector3 a a a 

magnitude :: Vector3 -> Float
magnitude (Vector3 x y z) = sqrt(x * x + y * y + z * z)

normalized :: Vector3 -> Vector3
normalized v = v * (uniformVector (1 / magnitude v))

rotate2D :: (Float, Float) -> Float -> (Float, Float)
rotate2D (x, y) angle = (cos(angle) * x - sin(angle) * y, sin(angle) * x + cos(angle) * y)

crossProduct :: Vector3 -> Vector3 -> Vector3
crossProduct (Vector3 x1 y1 z1) (Vector3 x2 y2 z2) = Vector3 (y1 * z2 - z1 * y2) (z1 * x2 - x1 * z2) (x1 * y2 - y1 * x2)

dotProduct :: Vector3 -> Vector3 -> Float
dotProduct (Vector3 x1 y1 z1) (Vector3 x2 y2 z2) = x1 * x2 + y1 * y2 + z1 * z2

type RayHit = (Surface, Ray, Float)

compRayHit :: RayHit -> RayHit -> Ordering
compRayHit (_, _, t1) (_, _, t2) = t1 `compare` t2

-- Infinite Plane : (point, outward_normal)
type Plane = (Vector3, Vector3) 

type Color = Vector3
darkGray :: Color
darkGray = Vector3 0.15 0.15 0.15

black :: Color
black = Vector3 0 0 0

sumColors :: Color -> Color -> Color
sumColors c1 c2 = c1 + c2

clampColor :: Color -> Color
clampColor (Vector3 r g b) = Vector3 (min (max r 0) 1) (min (max g 0) 1) (min (max b 0) 1)

data Surface = Triangle {v1::Vector3, v2::Vector3, v3::Vector3, n1::Vector3, n2::Vector3, n3::Vector3, rgb::Color, kr::Float, kd::Float, planeIn::Plane} deriving (Show, Eq)

interpolatedNormal :: Surface -> Vector3 -> Vector3
interpolatedNormal sf point =
  let d1 = magnitude (point - (v1 sf)) + 1e-6
      d2 = magnitude (point - (v2 sf)) + 1e-6
      d3 = magnitude (point - (v3 sf)) + 1e-6
      w1 = 1 / d1 
      w2 = 1 / d2
      w3 = 1 / d3
      nn = (((n1 sf) * (uniformVector w1)) + (n2 sf) * (uniformVector w2) + (n3 sf) * (uniformVector w3)) * uniformVector (1 / (w1 + w2 + w3))
  in normalized nn

type Lighting = (Vector3, Color)

type Ray = (Vector3, Vector3)

getRayPoint :: Ray -> Float -> Vector3
getRayPoint r t = let (origin, dir) = r in origin + (uniformVector t) * dir

{-
Section II
Parser-related functions

Parse objects placement, camera pose and lightings from a unity scene file (.unity)
Use Unity Editor for scene editing.
-}

-- https://wiki.haskell.org/Data.List.Split
splitOn :: (a -> Bool) -> [a] -> [[a]]
splitOn _ [] = []
splitOn f l@(x:xs)
  | f x = splitOn f xs
  | otherwise = let (h,t) = break f l in h:(splitOn f t)

parseVector3 :: String -> Vector3 -- x: 0, y: 0, z: 0, or r: 0, g:0.1, b:0.2
parseVector3 s = Vector3 (read $ spl !! 1::Float) (read $ spl !! 3::Float)  (read $ spl !! 5::Float)
                 where spl = splitOn (\c -> c == ':' || c == ',') s

parseFloat :: String -> Float
parseFloat s = read s::Float

parseFileId :: String -> String -- fileId: 12345
parseFileId = last . splitOn (== ' ') 

getPropertyLine :: [String] -> String -> String -- lines, property, propertyLine
getPropertyLine contents property = head $ filter (\l -> (take (length property) $ dropWhile (== ' ') l) == property) contents

data Component = Object {fileId::String, meshName::String} 
                | Camera {fileId::String}
                | Light {headerId::String, color::Color}
                | Transform {position::Vector3, eulerAngles::Vector3, scale::Vector3, headerId::String} 
                | Settings {color::Color, reflectivity::Float, diffuseness::Float, headerId::String} deriving Show

isObject :: Component -> Bool
isObject (Object _ name) = not $ isInfixOf "Light" name
isObject _ = False

isTransform :: Component -> Bool
isTransform (Transform _ _ _ _) = True
isTransform _ = False

isSettings :: Component -> Bool
isSettings (Settings _ _ _ _ ) = True
isSettings _ = False

isCamera :: Component -> Bool
isCamera (Camera _ ) = True
isCamera _ = False

isLight :: Component -> Bool
isLight (Light _ _) = True
isLight _ = False

parseSurface :: String -> [Surface]
-- leave planeIn unset
parseSurface l = let lns = lines l
                     unlabeledParse = \x -> let spl = splitOn (== ' ') x 
                                            in Vector3 (read $ spl !! 1::Float) (read $ spl !! 3::Float) (read $ spl !! 2::Float) 
                     -- [Vector3]
                     vcoords = map unlabeledParse $ filter (\ln -> take 2 ln == "v ") lns
                     vnormals = map unlabeledParse $ filter (\ln -> take 2 ln == "vn") lns
                     faces = filter (\ln -> take 1 ln == "f") lns
                     
                 in map (\f -> let  splitted = map (\c -> ((read c)::Int) - 1) (splitOn (\c -> (isSpace c) || c == '/' || c == 'f') f)
                                    vtnformat = (length splitted) == 9
                                    vnformat = (length splitted) == 6
                                    
                                    vcoord1 = splitted !! 0
                                    vnormal1
                                      | vtnformat = splitted !! 2
                                      | vnformat = splitted !! 1
                                      | otherwise = error ("Unsupported fbx format at " ++ f)
                                    vcoord2
                                      | vtnformat = splitted !! 3
                                      | vnformat = splitted !! 2
                                      | otherwise = error ("Unsupported fbx format at " ++ f)
                                    vnormal2 
                                      | vtnformat = splitted !! 5
                                      | vnformat = splitted !! 3
                                      | otherwise = error ("Unsupported fbx format at " ++ f)
                                    vcoord3
                                      | vtnformat = splitted !! 6
                                      | vnformat = splitted !! 4
                                      | otherwise = error ("Unsupported fbx format at " ++ f)
                                    vnormal3
                                      | vtnformat = splitted !! 8
                                      | vnformat = splitted !! 5
                                      | otherwise = error ("Unsupported fbx format at " ++ f)
                                    

                               in Triangle {v1 = vcoords !! vcoord1, v2 = vcoords !! vcoord2, v3 = vcoords !! vcoord3, 
                                            n1 = vnormals !! vnormal1, n2 = vnormals !! vnormal2, n3 = vnormals !! vnormal3,
                                            rgb = uniformVector 1, kd = 0, kr = 0, 
                                            planeIn = (uniformVector 0, uniformVector 0) } ) faces

-- reference: htrace.hs, http://www.nobugs.org/developer/htrace/index.html
make_pgm :: Integer -> Integer -> [ Color ] -> String
make_pgm width height xs = "P3\n" ++ show width ++ " " ++ show height ++ "\n255\n" ++ stringify(xs)
      where stringify [] = ""
            stringify ((Vector3 r g b):cs) = show (round (r*255)::Integer) ++ " " 
                    ++ show (round (g*255)::Integer) ++ " " 
                    ++ show (round (b*255)::Integer) ++ " " 
                    ++ stringify cs

{-
Section III
Geometry calculation helpers.
-}

-- transform local point represented in left-hand coordinate, to world point reprented in right-hand coordinate
-- Unity rotates in the order of Z-X-Y, so we should rotate in Y-X-Z
transformPoint :: Vector3 -> Component -> Vector3
transformPoint (Vector3 px py pz) (Transform (Vector3 posx posz posy) (Vector3 nax naz nay) (Vector3 scx scz scy) _) = 
  let psx = px * scx
      psy = py * scy
      psz = pz * scz 
      (pz' , px') = rotate2D (psz , psx) (-nay)
      (py' , pz'') = rotate2D (psy , pz') (-nax)
      (px'' , py'') = rotate2D (px' , py') (-naz)
  in (Vector3 posx posy posz) + (Vector3 px'' py'' pz'')
transformPoint _ _ = error "Invalid argument"

-- transform and normalize local vector to world
transformVectorNormal :: Vector3 -> Component -> Vector3
transformVectorNormal (Vector3 px py pz) (Transform _ (Vector3 nax naz nay) (Vector3 scx scz scy) _) =
  let psx = px * scx
      psy = py * scy
      psz = pz * scz
      (pz' , px') = rotate2D (psz , psx) (-nay)
      (py' , pz'') = rotate2D (psy , pz') (-nax)
      (px'' , py'') = rotate2D (px' , py') (-naz)
  in normalized $ (Vector3 px'' py'' pz'')
transformVectorNormal _ _ = error "Invalid argument"

parseSection :: [String] -> Maybe Component
parseSection (h1:h2:t)
  | take 6 h2 == "Light:" = Just Light {
                                headerId = parseFileId $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_GameObject",
                                color = parseVector3 $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_Color"}
  | take 10 h2 == "GameObject" && take 6 m_name == "Camera" = Just Camera {
                                fileId = last $ splitOn (== '&') h1}
  | take 10 h2 == "GameObject" = Just Object {
                                fileId = last $ splitOn (== '&') h1, 
                                meshName = last $ splitOn (== ' ') $ getPropertyLine t "m_Name"}

  | take 9 h2 == "Transform" = Just Transform{
                                position = parseVector3 $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_LocalPosition",
                                eulerAngles = (uniformVector deg2Rad) * (parseVector3 $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_LocalEulerAnglesHint"),
                                scale = parseVector3 $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_LocalScale",
                                headerId = parseFileId $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_GameObject"}
  | take 13 h2 == "MonoBehaviour" = Just Settings{
                                color = parseVector3 $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "color",
                                reflectivity = parseFloat $ last $ splitOn (== ' ') $ getPropertyLine t "reflectivity",
                                diffuseness = parseFloat $ last $ splitOn (== ' ') $ getPropertyLine t "diffuseness",
                                headerId = parseFileId $ last $ splitOn(\c -> c=='{' || c == '}') $ getPropertyLine t "m_GameObject"}
  where m_name = last $ splitOn (== ' ') $ getPropertyLine t "m_Name"
parseSection _ = Nothing


-- Rays in camera coordinate, not normalized yet
raysFromCamera :: Int -> Int -> Float -> [Ray]
raysFromCamera width height hFov = 
  let gridSize = 1 * tan(hFov * deg2Rad / 2) * 2 / (fromIntegral height)
      grids = [Vector3 (fromIntegral(x) * gridSize) 1 (fromIntegral(z) * gridSize) | z <- reverse [1..height], x <- [1..width]]
      vp = Vector3 (fromIntegral(width `div` 2) * gridSize) 0 (fromIntegral(height `div` 2) * gridSize)
  in map (\g -> (uniformVector 0, g - vp)) grids

-- (origin_x + dir_x * t - pp_x, origin_y + dir_y * t - pp_y, origin_z + dir_z * t - pp_z) `dot` (pn_x, pn_y, pn_z) = 0
rayPlaneIntersect :: Ray -> Plane -> Maybe Float
rayPlaneIntersect (origin, dir) (pp,pn)
  | dotProduct dir pn >= -1e-4 = Nothing
  | candidate_t < 1e-4 = Nothing          -- avoid intersection with origin surface
  | otherwise = Just candidate_t
  where t_coeff = dotProduct dir pn
        c = dotProduct (origin - pp) pn
        candidate_t = -c / t_coeff

-- point must be on the plane 
pointInTriangle :: Surface -> Vector3 -> Bool
pointInTriangle triangle point =
  let (_, pn) = planeIn triangle
      e12 = (v2 triangle) - (v1 triangle)
      e23 = (v3 triangle) - (v2 triangle)
      e31 = (v1 triangle) - (v3 triangle)
      side1 = dotProduct pn $ crossProduct e12 (point - v1 triangle)
      side2 = dotProduct pn $ crossProduct e23 (point - v2 triangle)
      side3 = dotProduct pn $ crossProduct e31 (point - v3 triangle)
  in (side1 >= 0 && side2 >= 0 && side3 >= 0) || (side1 <= 0 && side2 <= 0 && side3 <= 0)

raySurfaceIntersect :: Ray -> Surface -> Maybe RayHit
raySurfaceIntersect ray sf = 
  case rayPlaneIntersect ray (planeIn sf) of
      Nothing -> Nothing
      Just t -> case pointInTriangle sf (getRayPoint ray t) of
                    True -> Just (sf, ray, t)
                    False -> Nothing

-- reference: http://www.cs.utah.edu/~awilliam/box/box.pdfb
rayAABBIntersect :: Ray -> AABB -> Bool
rayAABBIntersect ((Vector3 ox oy oz), (Vector3 dirx diry dirz)) (lox, loy, loz, hix, hiy, hiz) = 
  let invdirx = 1 / dirx
      invdiry = 1 / diry
      invdirz = 1 / dirz
      tmin = case invdirx < 0 of
        True -> (hix - ox) * invdirx
        False -> (lox - ox) * invdirx
      tmax = case invdirx < 0 of
        True -> (lox - ox) * invdirx
        False -> (hix - ox) * invdirx
      tymin = case invdiry < 0 of 
        True -> (hiy - oy) * invdiry
        False -> (loy - oy) * invdiry
      tymax = case invdiry < 0 of
        True -> (loy - oy) * invdiry
        False -> (hiy - oy) * invdiry

      tmin' = max tmin tymin
      tmax' = min tmax tymax
      
      tzmin = case invdirz < 0 of
        True -> (hiz - oz) * invdirz
        False -> (loz - oz) * invdirz
      tzmax = case invdirz < 0 of
        True -> (loz - oz) * invdirz
        False -> (hiz - oz) * invdirz
      
      -- tmin' = max tmin tzmin
      tmax'' = min tmax' tzmax
       
  in not (tmin > tymax || tymin > tmax || tmin' > tzmax || tzmin > tmax' || tmax'' < 0)

{-
Section IV
Data Structure for query speed-up
-}
-- x_min, y_min, z_min, x_max, y_max, z_max
type AABB = (Float, Float, Float, Float, Float, Float)

-- depth partition_point lchild rchild
data KdtreeNode = Node [Surface] Int Float KdtreeNode KdtreeNode
                | Leaf [Surface] 

getSurfaceProperty :: Int ->  (Float -> Float -> Float -> Float) -> Surface -> Float
getSurfaceProperty axis f surface
  | axis == 0 = let Vector3 x1 _ _ = v1 surface
                    Vector3 x2 _ _ = v2 surface 
                    Vector3 x3 _ _ = v3 surface
                in f x1 x2 x3
  | axis == 1 = let Vector3 _ y1 _ = v1 surface
                    Vector3 _ y2 _ = v2 surface 
                    Vector3 _ y3 _ = v3 surface
                in f y1 y2 y3
  | axis == 2 = let Vector3 _ _ z1 = v1 surface
                    Vector3 _ _ z2 = v2 surface
                    Vector3 _ _ z3 = v3 surface
                in f z1 z2 z3
  | otherwise = error "wrong axis parameter"

getMax :: Ord a => a -> a -> a -> a
getMax a b c = maximum [a, b, c]
getMin :: Ord a => a -> a -> a -> a
getMin a b c = minimum [a, b, c]
getAvg :: Float -> Float -> Float -> Float
getAvg a b c = (a+b+c)/3

calculateAABB :: [Surface] -> AABB
calculateAABB sfs = 
  let minx = minimum $ map (\sf1 -> getSurfaceProperty 0 getMin sf1 ) sfs
      miny = minimum $ map (\sf1 -> getSurfaceProperty 1 getMin sf1 ) sfs
      minz = minimum $ map (\sf1 -> getSurfaceProperty 2 getMin sf1 ) sfs
      maxx = maximum $ map (\sf1 -> getSurfaceProperty 0 getMax sf1 ) sfs
      maxy = maximum $ map (\sf1 -> getSurfaceProperty 1 getMax sf1 ) sfs
      maxz = maximum $ map (\sf1 -> getSurfaceProperty 2 getMax sf1 ) sfs
  in (minx, miny, minz, maxx, maxy, maxz)                

buildKdtree :: [Surface] -> Int -> KdtreeNode
buildKdtree surfaces depth 
  | length(surfaces) < 3 = Leaf surfaces 
  | otherwise =
    let axis = depth `mod` 3
        cutPSum = sum $ map (getSurfaceProperty axis getAvg) surfaces
        cutP = cutPSum / (fromIntegral $ length surfaces)
        (leftChildren, nonLeft) = partition (\sf -> getSurfaceProperty axis getMax sf < cutP) surfaces
        (rightChildren, middle) = partition (\sf -> getSurfaceProperty axis getMin sf > cutP) nonLeft
    in Node middle depth cutP (buildKdtree leftChildren (depth + 1)) (buildKdtree rightChildren (depth + 1))

rayCast :: KdtreeNode -> AABB -> Ray -> [RayHit]
rayCast (Leaf surfaces) _ ray = catMaybes $ map (raySurfaceIntersect ray) surfaces
rayCast (Node surfaces depth cutPoint lchild rchild) (xmin, ymin, zmin, xmax, ymax, zmax) ray =
  let lchildAABB = case depth `mod` 3 of 
                        0 -> (xmin, ymin, zmin, cutPoint, ymax, zmax)
                        1 -> (xmin, ymin, zmin, xmax, cutPoint, zmax)
                        _ -> (xmin, ymin, zmin, xmax, ymax, cutPoint)
      rchildAABB = case depth `mod` 3 of
                        0 -> (cutPoint, ymin, zmin, xmax, ymax, zmax)
                        1 -> (xmin, cutPoint, zmin, xmax, ymax, zmax)
                        _ -> (xmin, ymin, cutPoint, xmax, ymax, zmax)
      (_, (Vector3 dirX dirY dirZ)) = ray
      rayFromMinToMax = case depth `mod` 3 of
                        0 -> dirX > 0
                        1 -> dirY > 0
                        _ -> dirZ > 0
      iflcHit = rayAABBIntersect ray lchildAABB
      ifrcHit = rayAABBIntersect ray rchildAABB

      lcHits = case iflcHit of
                  True -> rayCast lchild lchildAABB ray
                  False -> []
      rcHits = case ifrcHit of
                  True -> rayCast rchild rchildAABB ray
                  False -> []
      selfHits = catMaybes $ map (raySurfaceIntersect ray) surfaces

  in if rayFromMinToMax && (not (null lcHits)) 
        then selfHits ++ lcHits 
        else if (not (null rcHits)) then selfHits ++ rcHits else selfHits ++ lcHits ++ rcHits
       

globalLighting :: KdtreeNode -> AABB -> RayHit -> [Lighting] -> Int -> Color
globalLighting rootNode outmostAABB (sf, ray, t) lights depth =
  let origin = getRayPoint ray t
      originNormal = interpolatedNormal sf origin
      diffuseColor = map (\l -> let (lightPos, lightColor) = l
                                    rayFromLight = (lightPos, (normalized (origin - lightPos)))
                                    intersections = rayCast rootNode outmostAABB rayFromLight
                                    (hitSf, _, _) = minimumBy compRayHit intersections
                                    cd = max 0 $ dotProduct originNormal (normalized (lightPos - origin))
                                    mixedColor = (uniformVector (cd*(kd sf))) * (rgb sf) * lightColor

                                in if null intersections || (hitSf /= sf) then black else clampColor mixedColor
                            ) lights
      reflectedColor = let  (_, rayDir) = ray 
                            reflect = kr sf
                            normScale = -2 * (dotProduct originNormal rayDir)
                            reflectedDir = (uniformVector normScale) * originNormal + rayDir
                            nextIntersections = rayCast rootNode outmostAABB (origin, reflectedDir)
                            nextRayHit = minimumBy compRayHit nextIntersections
                            
                        in if reflect == 0 || depth == 0 || null nextIntersections then black else (uniformVector reflect) * (globalLighting rootNode outmostAABB nextRayHit lights (depth - 1))
  in clampColor (foldl sumColors (clampColor reflectedColor) diffuseColor)

{-
Section V
main function
-}
-- split a list into equal-length groups
multiSplit :: [a] -> Int -> [[a]]
multiSplit [] _ = [[]]
multiSplit l len
  | len == 1 = [l]
  | otherwise = let (ele, remaining) = splitAt (length l `div` len) l
                in ele : multiSplit remaining (len - 1)

main :: IO ()
main = do args <- getArgs
          case args of
            [sceneFileName, objDir, width, height] -> 
               do scene <- readFile sceneFileName
                  let linesOfScene = lines scene
                      separaters = filter (\l -> take 3 l == "---") linesOfScene
                      sectionsContents = drop 1 $ splitOn (\l -> take 3 l == "---") linesOfScene  --drop first 2 lines of headers
                      sections = zipWith (:) separaters sectionsContents  --[[String]]
                      parsedSections = catMaybes $ map parseSection sections  --[Component]
                      transforms = filter isTransform parsedSections
                      objects = filter isObject parsedSections
                      settings = filter isSettings parsedSections
                      mainCamera = head $ filter isCamera parsedSections
                      lits = filter isLight parsedSections
                  
                  objTexts <- mapM readFile $ map (\h -> objDir ++ (meshName h) ++ ".obj") objects  
                  
                  let
                      nameToMesh = zip (map meshName objects) (map parseSurface objTexts)  -- [("Cube", [Surface]]),..]
                      
                      surfacegroups = [sf | trans <- transforms, heads <- objects, sets <- settings, (meshN, rawMesh) <- nameToMesh,
                                  headerId trans == fileId heads, headerId sets == fileId heads, meshN == meshName heads,
                                  let sf = map (\s -> 
                                        let tv1 = transformPoint (v1 s) trans
                                            tv2 = transformPoint (v2 s) trans
                                            tv3 = transformPoint (v3 s) trans
                                            tn1 = transformVectorNormal (n1 s) trans
                                            tn2 = transformVectorNormal (n2 s) trans
                                            tn3 = transformVectorNormal (n3 s) trans
                                            pn_candidate = crossProduct (tv2 - tv1) (tv3 - tv1)
                                            pn | dotProduct pn_candidate (tn1 + tn2 + tn3) > 0 = pn_candidate  -- always point towards n's
                                                | otherwise = -pn_candidate

                                        in Triangle {v1 = tv1, v2 = tv2, v3 = tv3, 
                                                      n1 = tn1, n2 = tn2, n3 = tn3, 
                                                      rgb = color sets, kd = diffuseness sets, kr = reflectivity sets,
                                                      planeIn = (tv1, pn)}) 
                                            rawMesh ]
                  
                      surfaces = concat surfacegroups
                      kdTree = buildKdtree surfaces 0
                      outmostAABB = calculateAABB surfaces

                      lights = [(transformPoint (uniformVector 0) trans, color lit) | trans <- transforms, lit <- lits,
                            headerId trans == headerId lit]
                  let cameraTransform = head $ filter (\t -> headerId t == fileId mainCamera) transforms
                      cameraCoordRays = raysFromCamera (read width::Int) (read height::Int) 60
                      worldCoordRays = map (\r -> let (p, dir) = r
                                                      tp = transformPoint p cameraTransform
                                                      tdir = transformVectorNormal dir cameraTransform
                                                  in (tp, tdir)) cameraCoordRays
                                                 
                      solve = \ray -> let -- allHits = catMaybes $ map (raySurfaceIntersect ray) surfaces   -- Naive
                                          allHits = rayCast kdTree outmostAABB ray
                                          nearestHit = minimumBy compRayHit allHits  
                                      in if null allHits then darkGray else globalLighting kdTree outmostAABB nearestHit lights 2

                      raysPermuted = map (\i -> worldCoordRays !! ((10000019 * i) `mod` (length worldCoordRays))) [0..(length worldCoordRays)-1]
                      reverseMap = fromList $ map (\i -> (((10000019 * i) `mod` (length worldCoordRays)), i)) [0..(length worldCoordRays)-1]
                                      
                      (t0:t1:t2:t3:t4:t5:t6:t7:_) = multiSplit raysPermuted 8

                      solutions = runEval $ do
                        t0' <- rpar (force (map solve t0))
                        t1' <- rpar (force (map solve t1))
                        t2' <- rpar (force (map solve t2))
                        t3' <- rpar (force (map solve t3))
                        t4' <- rpar (force (map solve t4))
                        t5' <- rpar (force (map solve t5))
                        t6' <- rpar (force (map solve t6))
                        t7' <- rpar (force (map solve t7))
                        _ <- rseq t0'
                        _ <- rseq t1'
                        _ <- rseq t2'
                        _ <- rseq t3'
                        _ <- rseq t4'
                        _ <- rseq t5'
                        _ <- rseq t6'
                        _ <- rseq t7'
                        return (t0' ++ t1' ++ t2' ++ t3' ++ t4' ++ t5' ++ t6' ++ t7') 
                      
                      solutionsRestored = map (\i -> solutions !! (reverseMap ! i)) [0..(length worldCoordRays)-1]

                      -- solutionsRestored = map solve worldCoordRays `using` parList rdeepseq
                  
                  writeFile "out.pgm" ( (make_pgm (read width::Integer) (read height::Integer) solutionsRestored))
           
            _   -> do pn <- getProgName
                      die $ "Usage: " ++ pn ++ " <*.unity> <obj_directory> <width> <height>"
