{
module RenParse (RendDesc(RendDesc), readScene) where

import Tracer (Scene(Scene), Texture(Texture), Diffuse(Solid), TexturedObj, 
		   Light(AmbientL, PointL), Camera(Camera))
import RayMath (Dimension, Resolution, RenderObj(Sphere,Plane))
import Data.Char (isSpace, isAlpha, isDigit)


--scene which should be rendered
data RendDesc = RendDesc Resolution Int Scene


readScene:: String -> RendDesc
readScene= parseScene.lexer

}

%name parseScene
%tokentype { Token }

%token 
      int              { TokenInt $$ }
      double           { TokenDouble $$ }
      camera           { TokenCamera }
      background       { TokenBackground }
      diffuse          { TokenDiffuse }
      solid            { TokenSolid }
      texture          { TokenTexture }
      sphere           { TokenSphere }  
      plane            { TokenPlane }  
      obj              { TokenTexturedObj }
      objs             { TokenObjs }
      pointL           { TokenPointL }
      ambientL         { TokenAmbientL }
      lights           { TokenLights }
      scene            { TokenScene }
      resolution       { TokenResolution }
      rendDesc         { TokenRendDesc }
      '{'              { TokenOpenAcc }
      '}'              { TokenCloseAcc }
      '('              { TokenOpenBrack }
      ')'              { TokenCloseBrack }
      '['              { TokenOpenHook }
      ']'              { TokenCloseHook }
      ','              { TokenComma }

%%

RendDesc : rendDesc Resolution int '[' Scene ']'                         { RendDesc $2 $3 $5}

Resolution  : '(' resolution '(' int ',' int ')' ')'                           { ($4,$6) }

Scene       : scene '{' Camera '}' '{' Background '}' 
                    '{' objs Objs '}' '{' lights Lights '}'              { Scene $3 $6 $10 $14 }

Camera      : camera '(' double ',' double ',' double ')' 
                     '(' int ',' int ')'                                       { Camera ($3,$5,$7) ($10,$12) }

Background  : background '(' double ',' double ',' double ')'                  { ($3,$5,$7) }

Obj      : sphere double '(' double ',' double ',' double ')'               { Sphere $2 ($4,$6,$8) }
            | plane '(' double ',' double ',' double ',' double ')'            { Plane ($3,$5,$7,$9) }

Diffuse        : solid '(' double ',' double ',' double ')'                       { Solid ($3,$5,$7) }

Texture    : '(' texture '(' diffuse Diffuse ')' double int double double')'         { Texture $5 $7 $8 $9 $10 }

TexturedObj : '(' obj '(' Obj ')' Texture ')'                         { ($4,$6) }

Objs     : {- empty -}                                                      { [] }
            | Objs TexturedObj                                           { $2 : $1 }

Light       : '(' pointL '(' double ',' double ',' double ')' 
                            '(' double ',' double ',' double ')' ')'           { PointL ($4,$6,$8) ($11,$13,$15) }
            | '(' ambientL '(' double ',' double ',' double ')' ')'        { AmbientL ($4,$6,$8) }

Lights      : {- empty -}                                                      { [] }
            | Lights Light                                                     { $2 : $1 }

{

happyError :: [Token] -> a
happyError _ = error "Parse error ! ! !"

data Token
       = TokenInt Int
       | TokenDouble Double
       | TokenCamera
       | TokenBackground
       | TokenDiffuse
       | TokenSolid
       | TokenTexture
       | TokenSphere
       | TokenPlane
       | TokenObjType
       | TokenTexturedObj
       | TokenObjs
       | TokenPointL
       | TokenAmbientL
       | TokenLight
       | TokenLights
       | TokenScene
       | TokenResolution
       | TokenRendDesc
       | TokenOpenAcc
       | TokenCloseAcc
       | TokenOpenBrack
       | TokenCloseBrack
       | TokenOpenHook
       | TokenCloseHook
       | TokenComma

lexer :: String -> [Token]
lexer [] = []
lexer (c:cs) 
      | isSpace c = lexer cs
      | isAlpha c = lexVar (c:cs)
      | isDigit c = lexNum (c:cs) 1
lexer ('{':cs) = TokenOpenAcc : lexer cs
lexer ('}':cs) = TokenCloseAcc : lexer cs
lexer ('(':cs) = TokenOpenBrack : lexer cs
lexer (')':cs) = TokenCloseBrack : lexer cs
lexer ('[':cs) = TokenOpenHook : lexer cs
lexer (']':cs) = TokenCloseHook : lexer cs
lexer (',':cs) = TokenComma : lexer cs
lexer ('-':cs) = lexNum cs (-1)

lexNum cs mul
          | (r == '.')  = TokenDouble (mul * (read (num++[r]++num2) :: Double)) : lexer rest2
          | otherwise = TokenInt (round (mul * (read num))) : lexer (r:rest)
          where (num,(r:rest)) = span isDigit cs
	        (num2,rest2)   = span isDigit rest

lexVar cs =
   case span isAlpha cs of
      ("camera",rest)            -> TokenCamera : lexer rest
      ("background",rest)        -> TokenBackground : lexer rest
      ("diffuse",rest)           -> TokenDiffuse : lexer rest
      ("solid",rest)             -> TokenSolid : lexer rest
      ("texture",rest)           -> TokenTexture : lexer rest
      ("sphere",rest)            -> TokenSphere : lexer rest
      ("plane",rest)             -> TokenPlane : lexer rest
      ("obj",rest)            -> TokenTexturedObj : lexer rest
      ("objs",rest)           -> TokenObjs : lexer rest
      ("pointL",rest)        -> TokenPointL : lexer rest
      ("ambientL",rest)          -> TokenAmbientL : lexer rest
      ("light",rest)             -> TokenLight : lexer rest
      ("lights",rest)            -> TokenLights : lexer rest
      ("scene",rest)             -> TokenScene : lexer rest
      ("resolution",rest)        -> TokenResolution : lexer rest
      ("rendDesc",rest)          -> TokenRendDesc : lexer rest
  }
