module Submission where
import System.Random.MWC
import Control.Monad
import Control.Monad.Par
import Control.Applicative ((<$>))
import Statistics.Sample
import Data.List
import Data.Vector(fromList)
import qualified Statistics.Distribution as D
import Statistics.Distribution.Normal

data GraphParameters = GraphParameters {
        numNodes :: Int,
        edges :: [(Int,Int)]
    } deriving (Eq, Show)

numEdges :: GraphParameters -> Int
numEdges = length . edges

erdosRenyiGraph' :: Int -> Double -> IO ()
erdosRenyiGraph' n p = withSystemRandom . asGenIO $ \gen -> erdosRenyiGraph gen n p

erdosRenyiGraph :: GenIO -> Int -> Double -> IO ()
erdosRenyiGraph gen n p = do
    let allNodes = [0..(n-1)]
    let outgoingEdgeTargets = selectWithProbability gen p allNodes
    let singleNodeEdges node = zip (repeat node) <$> outgoingEdgeTargets
    allEdges <- concat <$> mapM singleNodeEdges allNodes
    let graph = GraphParameters (length allNodes) allEdges
    putStrLn "Graph analysis"
    putStrLn ("Erdos Renyi Graph with " ++ (show (length allNodes)) ++ " nodes and " ++ (show (length allEdges)) ++ " edges")
    analyzeDegrees $ degreeDistribution graph

selectWithProbability :: GenIO -> Double -> [a] -> IO [a]
selectWithProbability _   _ [] = return []
selectWithProbability gen p (x:xs) = do
    r <- uniform gen :: IO Double
    let v = [x | r <= p]
    liftM2 (++) (return v) $ selectWithProbability gen p xs

degreeDistribution :: GraphParameters -> [Int]
degreeDistribution graph = map length . group $ sort $ (concat [[a,b] | (a, b) <- (edges graph)])

-- Returns Mean, Median, Mode, Stddev for degrees of nodes
analyzeDegrees :: [Int] -> IO ()
analyzeDegrees degrees = putStrLn ("Mean degree is " ++ (show (mean sample)) ++ " and the distribution has kurtosis " ++ (show (kurtosis sample)))
    where sample = fromList $ map fromIntegral degrees

main = do
    erdosRenyiGraph' 10 0.1
    erdosRenyiGraph' 100 0.1
    erdosRenyiGraph' 1000 0.1
    erdosRenyiGraph' 10000 0.1
{-

 Name: Dhruv Singh
 Uni: ds3638

 Collaborators:

 References: https://github.com/ulikoehler/graph-generators/

 ------------------------------

 COMS 4995 003 Parallel Functional Programming

 Project

 An infinite graph generator with report on inputted metric function

 Your code should load under GHCi 8.6.5 with no warnings under -Wall, e.g.
 :set -Wall
 :l hw4

-}