{-
3D SCENE

This module does the initial work of loading in the 3D data
and parsing it. 
-}

module Scene
  ( fromPLY
  , projectScene
  )
where

import           PLY.Types
import           Lib
import           PLY
import           Data.Array
import           Data.Vector
import           Data.Map.Strict                ( Map
                                                , empty
                                                , insertWith
                                                , findWithDefault
                                                )
import qualified Data.Cross                    as Vec
import           Data.Maybe
import           Data.ByteString.Char8
import           Projection                     ( Projection(..)
                                                , Vertex2(..)
                                                )
import           Control.Parallel.Strategies

data Coor3D = Coor3D (Float, Float, Float) deriving (Show)
type Face = [Vertex3]
type NormalMap = Map Vertex3 (Vec.Three Float)

data Vertex3 = Vertex3 { coor3 :: Vec.Three Float
                     , rgb3 :: Rgb
                     } deriving (Show, Eq,  Ord)
data Scene = Scene { vertexList3 :: [Vertex3]
                   , faceList3 :: [Face]
                   , light3 :: Vec.Three Float
                   , focalDist :: Float
                   } deriving (Show)

fromPLY :: [Char] -> Vec.Three Float -> Vec.Three Float -> Float -> IO Scene
fromPLY filename o lightVec focalD = do
  vertexData <- parseVertex3Ply
    (loadElements (Data.ByteString.Char8.pack "vertex") filename)
  faceData <- parseFacePly
    (loadElements (Data.ByteString.Char8.pack "face") filename)
  return $ buildScene vertexData faceData o lightVec focalD

buildScene
  :: [Vertex3]
  -> [[Int]]
  -> Vec.Three Float
  -> Vec.Three Float
  -> Float
  -> Scene
buildScene vertexData faceData o lightVec focalD = Scene { vertexList3 = vList
                                                         , faceList3   = fList
                                                         , light3 = lightVec
                                                         , focalDist   = focalD
                                                         }
 where
  vList = chunkPar (translateOrigin o) vertexData
  fList = constructFaceList vList faceData


translateOrigin :: Vec.Three Float -> Vertex3 -> Vertex3
translateOrigin ((ox, oy, oz)) Vertex3 { coor3 = (x, y, z), rgb3 = rgb3' } =
  Vertex3 { coor3 = (x - ox, y - oy, z - oz), rgb3 = rgb3' }


constructVertex3Vector :: Vector (Vector Scalar) -> [Vertex3]
constructVertex3Vector vs =
  catMaybes $ chunkPar constructVertex3 (Data.Vector.toList vs)

unpackFaceIndices :: Vector (Vector Scalar) -> [[Int]]
unpackFaceIndices faces = chunkPar
  (\x -> ((chunkPar unpackInt) . Data.Vector.toList) x)
  (Data.Vector.toList faces)
 where
  unpackInt (Sint x) = x
  unpackInt _        = error "Invalid face data"

parseVertex3Ply :: IO (Either String (Vector (Vector Scalar))) -> IO [Vertex3]
parseVertex3Ply input = do
  elements <- input
  case elements of
    Left  _   -> error "Invalid input"
    Right dat -> return $ constructVertex3Vector dat

parseFacePly :: IO (Either String (Vector (Vector Scalar))) -> IO [[Int]]
parseFacePly input = do
  elements <- input
  case elements of
    Left  _   -> error "Invalid input"
    Right dat -> return $ unpackFaceIndices dat


constructFaceList :: [Vertex3] -> [[Int]] -> [Face]
constructFaceList vList faceData = chunkPar
  (chunkPar ((Data.Array.!) vertexArr))
  faceData
 where
  vertexArr = array (0, Prelude.length vList - 1) (Prelude.zip [0 ..] vList)

constructVertex3 :: Vector Scalar -> Maybe Vertex3
constructVertex3 v = case v' of
  -- Vertex3 with full color data
  [Sfloat x', Sfloat y', Sfloat z', Suchar r', Suchar g', Suchar b'] ->
    Just $ Vertex3 { coor3 = (x', y', z'), rgb3 = (r', g', b') }
  -- Vertex3 with no color data, default RGB3 value to (0,0,0)
  [Sfloat x', Sfloat y', Sfloat z'] ->
    Just $ Vertex3 { coor3 = (x', y', z'), rgb3 = (0, 0, 0) }
  -- Invalid vertex data
  _ -> Nothing
  where v' = Data.Vector.toList v



computeFaceNormal :: Face -> Vec.Three Float
computeFaceNormal (a : b : c : _) = computeNormal a b c
 where
  computeNormal v0 v1 v2 =
    Vec.cross3 (computeDifference v0 v1) (computeDifference v0 v2)
  computeDifference Vertex3 { coor3 = (x0, y0, z0) } Vertex3 { coor3 = (x1, y1, z1) }
    = (x1 - x0, y1 - y0, z1 - z0)
computeFaceNormal _ = error "Not enough vertices in face spec"


vertexNormalMap :: [Face] -> NormalMap
vertexNormalMap fs = Prelude.foldl (insertWith') Data.Map.Strict.empty fs
 where
  insertWith' hm face = Prelude.foldl insertNormal hm face   where
    insertNormal hm' v =
      Data.Map.Strict.insertWith (\new old -> addVec new old) v faceNormal hm'
    faceNormal = computeFaceNormal face


projectVertex3 :: Float -> NormalMap -> Vertex3 -> Vertex2
projectVertex3 dist m v@Vertex3 { coor3 = c@(x, y, z), rgb3 = rgb } = Vertex2
  { coor2  = (x * fd, z * fd)
  , depth  = vecLen c
  , normal = normalize $ Data.Map.Strict.findWithDefault (0, 0, 0) v m
  , rgb2   = rgb
  }
  where fd = dist / y


{- Mapping from 3D to 2D while retaining necessary information -}
projectScene :: Scene -> Projection
projectScene Scene { faceList3 = fs, focalDist = dist, light3 = lit } =
  Projection { vertexList2 = Prelude.concat fs', faceList2 = fs', light2 = lit }
 where
  vNorms = vertexNormalMap fs
  fs'    = chunkPar (Prelude.map (projectVertex3 dist vNorms)) fs



