{-
2D PROJECTION INFO

This tracks the information of the 2D projection of our scene in order
to properly calculate the pixel display.
-}

module Projection
  ( Projection(..)
  , Vertex2(..)
  , genPixels
  )
where

import qualified Data.Cross                    as Vec
import           Data.Word                      ( Word8 )
import           Control.Parallel.Strategies
import           Lib
import           PixelDraw
import           HSE


type Face = [Vertex2]

data Vertex2 = Vertex2 { coor2 :: Vec.Two Float
                       , depth :: Float
                       , normal :: Vec.Three Float
                       , rgb2 :: Rgb
                       } deriving (Show, Eq,  Ord)
data Projection = Projection { vertexList2 :: [Vertex2]
                   , faceList2 :: [Face]
                   , light2 :: Vec.Three Float
                   }

minMaxCoors :: [Vertex2] -> (Float, Float, Float, Float)
minMaxCoors [] = error "Not enough vertices"
minMaxCoors (Vertex2 { coor2 = (x, y) } : tl) = minMaxCoor' (x, y, x, y) tl
 where
  minMaxCoor' (minx, miny, maxx, maxy) (Vertex2 { coor2 = (x', y') } : tl') =
    minMaxCoor' (min minx x', min miny y', max maxx x', max maxy y') tl'
  minMaxCoor' res [] = res


faceCoors2Pixels :: Int -> [Face] -> [[Pixel]]
faceCoors2Pixels numPixels fs = parMap rpar (parMap rpar f) fs
 where
  (minX, minY, maxX, maxY) = minMaxCoors $ Prelude.concat fs
  pixelWidth = ((maxX + 1) - (minX - 1)) / (fromIntegral numPixels)
  f (Vertex2 { coor2 = (x, y), depth = d, normal = n, rgb2 = rgb }) = Pixel
    { pX     = (round ((x - minX + 1) / pixelWidth))
    , pY     = (round ((y - minY + 1) / pixelWidth))
    , pDepth = d
    , pRgb   = rgb
    , pNorm  = n
    }


shadePixel :: Projection -> Pixel -> Pixel
shadePixel (Projection { light2 = light }) Pixel { pX = x, pY = y, pDepth = d, pRgb = (r, g, b), pNorm = norm }
  = Pixel { pX     = x
          , pY     = y
          , pDepth = d
          , pRgb   = snap (multRgb r, multRgb g, multRgb b)
          , pNorm  = norm
          }
 where
  intensity = abs $ (negDotProd light norm) / ((vecLen light) * (vecLen norm))
  multRgb rgbVal = fromIntegral (round val)
    where val = (fromIntegral (toInteger rgbVal)) * intensity



{-
Main function that generates all pixels to be drawn given a 2D projection spec
-}
genPixels :: Projection -> Int -> [Pixel]
genPixels proj@(Projection { faceList2 = fs }) screenW = parMap
  rpar
  (shadePixel proj)
  uniquePixels
 where
  uniquePixels =
    HSE.removeHiddenPixels filledInPixels $ Prelude.concat outlines
  facePixels     = faceCoors2Pixels screenW fs
  outlines       = parMap rpar genSegments facePixels
  filledInPixels = Prelude.concat $ parMap rpar getFacePixels outlines
