{-
PIXEL DRAWING PROCEDURES

These functions take care of properly tracing lines of pixels
and doing scan-lines of surfaces.
-}

module PixelDraw
  ( getFacePixels
  , genSegments
  , linePixels
  )
where

import           Data.Map.Strict                ( lookup
                                                , insert
                                                , empty
                                                , elems
                                                )
import           Control.Parallel.Strategies
import           Lib

getFacePixels :: [Pixel] -> [Pixel]
getFacePixels border = Prelude.concat $ (chunkPar) linePixels scanLines
 where
  scanLines =
    Data.Map.Strict.elems $ Prelude.foldl f Data.Map.Strict.empty border
  f hm p = entryUpdate p hm
  entryUpdate newP@(Pixel { pX = x, pY = y }) hm = Data.Map.Strict.insert
    y
    (newMin, newMax)
    hm
   where
    (newMin, newMax) = case (Data.Map.Strict.lookup y hm) of
      Nothing -> (newP, newP)
      Just (oldMinP@(Pixel { pX = oldMin }), oldMaxP@(Pixel { pX = oldMax }))
        -> if x < oldMin
          then (newP, oldMaxP)
          else if x > oldMax then (oldMinP, newP) else (oldMinP, oldMaxP)

genSegments :: [Pixel] -> [Pixel]
genSegments facePixels = Prelude.concat $ chunkPar linePixels (segs facePixels)
  where segs a = Prelude.zip a $ Prelude.tail $ cycle a


linePixelsLow :: (Pixel, Pixel) -> [Pixel]
linePixelsLow (p0@Pixel { pX = x0, pY = y0, pDepth = d0, pRgb = rgb0, pNorm = norm0 }, p1@Pixel { pX = x1, pY = y1, pDepth = d1, pRgb = rgb1, pNorm = norm1 })
  = if dx == 0
    then (straightVertLine (p0, p1))
    else linePixelsLow' d0 rgb0 norm0 dStart yStart [x0 .. x1]
 where
  dz       = abs $ (d1 - d0) / dist
  dx       = x1 - x0
  dRgb     = rgbDelta rgb0 rgb1 dist
  dVec     = vecDelta norm0 norm1 dist
  dist     = (fromIntegral (x1 - x0))

  (yi, dy) = if y1 < y0 then ((-1), (-1) * (y1 - y0)) else (1, y1 - y0)
  dStart   = (2 * dy) - dx
  yStart   = y0

  linePixelsLow' _ _ _ _ _ [] = []
  linePixelsLow' z rgb norm d y (x : tl) =
    (Pixel { pX = x, pY = y, pDepth = z, pRgb = rgb, pNorm = norm })
      : linePixelsLow' (z + dz) (incRgb rgb dRgb) (incVec norm dVec) d' y' tl
   where
    (y', d') =
      if d > 0 then (y + yi, d - (2 * dx) + (2 * dy)) else (y, d + (2 * dy))

linePixelsHigh :: (Pixel, Pixel) -> [Pixel]
linePixelsHigh (p0@Pixel { pX = x0, pY = y0, pDepth = d0, pRgb = rgb0, pNorm = norm0 }, p1@Pixel { pX = x1, pY = y1, pDepth = d1, pRgb = rgb1, pNorm = norm1 })
  = if dx == 0
    then (straightVertLine (p0, p1))
    else linePixelsHigh' d0 rgb0 norm0 dStart xStart [y0 .. y1]
 where
  dz       = abs $ (d1 - d0) / dist
  dy       = y1 - y0
  dRgb     = rgbDelta rgb0 rgb1 dist
  dVec     = vecDelta norm0 norm1 dist
  dist     = (fromIntegral (y1 - y0))

  (xi, dx) = if x1 < x0 then ((-1), (-1) * (x1 - x0)) else (1, x1 - x0)
  dStart   = (2 * dx) - dy
  xStart   = x0

  linePixelsHigh' _ _ _ _ _ [] = []
  linePixelsHigh' z rgb norm d x (y : tl) =
    (Pixel { pX = x, pY = y, pDepth = z, pRgb = rgb, pNorm = norm })
      : linePixelsHigh' (z + dz) (incRgb rgb dRgb) (incVec norm dVec) d' x' tl
   where
    (x', d') =
      if d > 0 then (x + xi, d - (2 * dy) + (2 * dx)) else (x, d + (2 * dx))

straightVertLine :: (Pixel, Pixel) -> [Pixel]
straightVertLine (Pixel { pX = x, pY = y0, pDepth = d0, pRgb = rgb0, pNorm = norm0 }, Pixel { pY = y1, pDepth = d1, pRgb = rgb1, pNorm = norm1 })
  = straightVertLine' d0 rgb0 norm0 [y0 .. y1] where
  dz   = abs $ (d1 - d0) / dist
  dRgb = rgbDelta rgb0 rgb1 dist
  dVec = vecDelta norm0 norm1 dist
  dist = (fromIntegral (y1 - y0))
  -- interpolate color data and depth data along line
  straightVertLine' _ _ _ [] = []
  straightVertLine' z rgb norm (hd : tl) =
    (Pixel { pX = x, pY = hd, pDepth = z, pRgb = rgb, pNorm = norm })
      : straightVertLine' (z + dz) (incRgb rgb dRgb) (incVec norm dVec) tl

linePixels :: (Pixel, Pixel) -> [Pixel]
linePixels (p0@Pixel { pX = x0, pY = y0 }, p1@Pixel { pX = x1, pY = y1 }) =
  xyLine
 where
  xyLine = if (abs (y1 - y0)) < (abs (x1 - x0))
    then if x0 > x1 then linePixelsLow (p1, p0) else linePixelsLow (p0, p1)
    else if y0 > y1 then linePixelsHigh (p1, p0) else linePixelsHigh (p0, p1)
