{-
CONVENIENCE FUNCTIONS

This module holds several convenience functions for various necessary
calculations.
-}


module Lib
  ( Pixel(..)
  , Rgb(..)
  , vecLen
  , incVec
  , incRgb
  , vecDelta
  , snap
  , negDotProd
  , rgbDelta
  , normalize
  , addVec
  , chunkPar
  )
where
import           Data.Word                      ( Word8 )
import qualified Data.Cross                    as Vec
import           Control.Parallel.Strategies



type Rgb = (Word8, Word8, Word8)
data Pixel = Pixel { pX :: Int
                   , pY :: Int
                   , pDepth :: Float
                   , pRgb :: Rgb
                   , pNorm :: (Vec.Three Float)
                   } deriving (Show)



incRgb :: Rgb -> (Float, Float, Float) -> Rgb
incRgb ((w0, w1, w2)) (dr, dg, db) = (addRgb w0 dr, addRgb w1 dg, addRgb w2 db)
 where
  addRgb rgbVal delta = fromIntegral (round fVal)
    where fVal = (fromIntegral (toInteger rgbVal)) + delta

vecDelta :: Vec.Three Float -> Vec.Three Float -> Float -> Vec.Three Float
vecDelta (x0, y0, z0) (x1, y1, z1) dist =
  ((x1 - x0) / dist, (y1 - y0) / dist, (z1 - z0) / dist)

incVec :: Vec.Three Float -> Vec.Three Float -> Vec.Three Float
incVec (x, y, z) (dx, dy, dz) = normalize (x + dx, y + dy, z + dz)


normalize :: Vec.Three Float -> Vec.Three Float
normalize (x, y, z) = (x / vLen, y / vLen, z / vLen)
  where vLen = vecLen (x, y, z)

vecLen :: Vec.Three Float -> Float
vecLen (x, y, z) = sqrt ((x ** 2) + (y ** 2) + (z ** 2))

snap :: Rgb -> Rgb
snap ((r, g, b)) = (snapVal r, snapVal g, snapVal b)
 where
  snapVal rgbVal = fromIntegral (min 255 (max 0 val))
    where val = (fromIntegral (toInteger rgbVal))

negDotProd :: Vec.Three Float -> Vec.Three Float -> Float
negDotProd (x0, y0, z0) (x1, y1, z1) =
  (-1) * ((x0 * x1) + (y0 * y1) + (z0 * z1))

rgbDelta :: Rgb -> Rgb -> Float -> (Float, Float, Float)
rgbDelta ((r0, g0, b0)) ((r1, g1, b1)) dist =
  (wordDiffDiv r0 r1 dist, wordDiffDiv g0 g1 dist, wordDiffDiv b0 b1 dist)
 where
  wordDiffDiv w0 w1 d =
    (fromIntegral ((toInteger w1) - (toInteger w0)) :: Float) / dist

addVec :: Vec.Three Float -> Vec.Three Float -> Vec.Three Float
addVec (x0, y0, z0) (x1, y1, z1) = (x0 + x1, y0 + y1, z0 + z1)


chunkedParMap :: Int -> Strategy b -> (a -> b) -> [a] -> [b]
chunkedParMap n strat f = withStrategy (parListChunk n strat) . map f

chunkPar = chunkedParMap 1000 rpar


