import Control.Parallel(par)
import Data.Colour.SRGB.Linear hiding (blend)
import Data.Colour hiding (blend)
import Data.Colour.RGBSpace
import Data.Colour.RGBSpace.HSV
import Data.IORef
import Graphics.GLUtil
import Graphics.GLUtil.JuicyTextures
import Graphics.Rendering.OpenGL
import Graphics.UI.GLUT
import System.Environment
import System.Random
import System.Exit

data Particle = Particle { colorHue :: GLfloat
                         , colorSaturation :: GLfloat
                         , colorValue :: GLfloat
                         , size :: GLfloat
                         , x :: GLfloat
                         , y :: GLfloat
                         , z :: GLfloat
                         , xVel :: GLfloat
                         , yVel :: GLfloat
                         , zVel :: GLfloat
                         , time :: Int
                         } deriving (Eq, Show, Read)

type ParticlesArr = [Particle]

grav = 0.0015 :: GLfloat
simulationSize = Size 900 900

main :: IO ()
main = do
  args <- getArgs
  case args of
    [pic,time,num_p] -> do
        let run_time = read time
        let num_particles = read num_p
        (_progName, _args) <- getArgsAndInitialize
        particles <- newIORef ([] :: ParticlesArr)
        color <- newIORef (0.0 :: GLfloat)
        window <- createWindow "Parallel Functional Programming Final Project"
        windowSize $= simulationSize
        displayCallback $= (displaySimulation particles)
        reshapeCallback $= Just reshape
        keyboardMouseCallback $= Just keyboardMouse
        idleCallback $= Just (idleProcess run_time num_particles particles color)
        case pic of
            "stephen" -> do
                let picture = "stephenpic.png"
                spritetex <- getTexture picture
                mainLoop
            "particle" -> do
                let picture = "particle.png"
                spritetex <- getTexture picture
                mainLoop
            _ -> error "Picture type must be either 'particle' or 'stephen'"
    _ -> error "Usage:  simulation <pic type> <run time> <number of particles>"

reshape s@(Size w h) = do
  viewport $= (Position 0 0, s)
  postRedisplay Nothing

keyboardMouse (Char '\ESC') Down _ _ = exitSuccess
keyboardMouse (Char '\BS') Down _ _ = exitSuccess
keyboardMouse k s m p = putStrLn $ show k

extract :: (IO (Either String a)) -> IO a
extract action = do
  ioAction <- action
  case ioAction of
    Left a -> error a
    Right b -> return b

displayParticle :: Particle -> IO ()
displayParticle p = do
  color $ findColor (colorHue p) (colorSaturation p) (colorValue p)
  texCoord $ TexCoord2 zero zero
  vertex $ Vertex3 (posx - psize) (posy - psize) (posz)
  texCoord $ TexCoord2 zero one
  vertex $ Vertex3 (posx - psize) (posy + psize) (posz)
  texCoord $ TexCoord2 one one
  vertex $ Vertex3 (posx + psize) (posy + psize) (posz)
  texCoord $ TexCoord2 one zero
  vertex $ Vertex3 (posx + psize) (posy - psize) (posz)
  where
    posx = x p
    posy = y p
    posz = z p
    psize = size p
    zero = 0 :: GLfloat
    one = 1 :: GLfloat

findColor :: GLfloat -> GLfloat -> GLfloat -> Color3 GLfloat
findColor hue sat val = Color3 (par chanr (chanr)) (par chang (chang)) (par chanb (chanb))
  where
    rgbchannel = (hsv hue sat val)
    chanr = (channelRed rgbchannel)
    chang = (channelGreen rgbchannel)
    chanb = (channelBlue rgbchannel)

newColor val
  | val > 360 = par newVal (newVal)
  | otherwise = val
    where
      newVal = newColor (val - 360)

getTexture pic = do
  picTexture <- extract (readTexInfo pic loadTexture)
  texture Texture2D $= Enabled
  activeTexture $= TextureUnit 0
  textureBinding Texture2D $= Just picTexture
  textureFilter Texture2D $= ((Linear', Just Nearest), Linear')
  textureWrapMode Texture2D S $= (Mirrored, ClampToEdge)
  textureWrapMode Texture2D T $= (Mirrored, ClampToEdge)
  blend $= Enabled
  blendFunc $= (SrcAlpha, OneMinusSrcAlpha)
  generateMipmap' Texture2D
  return picTexture

displaySimulation :: IORef ParticlesArr -> IO ()
displaySimulation particleArr = do
  clear [ColorBuffer]
  showParticles <- get particleArr
  mapM_ (renderPrimitive Quads . displayParticle) showParticles
  flush
  postRedisplay Nothing

idleProcess :: GLfloat -> Int -> IORef ParticlesArr -> IORef GLfloat -> IO ()
idleProcess run_time num_particles particles curr_color = do
        curr_particles <- get particles
        let moveParticles = map moveParticle curr_particles
        initParticles <- sequence (replicate num_particles (initParticle curr_color))
        let totalParticles = (initParticles ++ moveParticles)
        particles $= filter (\particle -> time particle >= 0)  totalParticles
        colorVal <- get curr_color
        curr_color $= newColor (colorVal + 0.2)
        if colorVal > run_time
            then do exitSuccess
            else do putStrLn (show colorVal)

moveParticle :: Particle -> Particle
moveParticle p =
  p { yVel = par vel (vel - grav)
    , x = par posx (posx + xVel p)
    , y = par posy (posy + yVel p)
    , z = par posz (posz + zVel p)
    , time = par ttl (ttl - 1)
    , colorSaturation = par sat (sat * 1.3)
    , colorValue = par val (val - 0.015)
    , size = par psize (psize * 1.02)
    }
        where
            vel = yVel p
            posx = x p
            posy = y p
            posz = z p
            ttl = time p
            sat = colorSaturation p
            val = colorValue p
            psize = size p

initParticle :: IORef GLfloat -> IO Particle
initParticle huebase = do
  base <- get huebase
  phue <- randomRIO (base, base + 20)
  psaturation <- randomRIO (0.0001, 0.15 :: GLfloat)
  xv <- randomRIO (-0.04, 0.04)
  yv <- randomRIO (0.03, 0.065)
  return $ Particle { x = 0
                    , y = -1
                    , z = 1
                    , colorHue = newColor phue
                    , colorSaturation = psaturation
                    , colorValue = 1
                    , size = 0.04
                    , xVel = xv
                    , yVel = yv
                    , zVel = 0
                    , time = 60
                    }
