import Data.List
import Data.Maybe
import qualified Data.Map as Map

data Chess = Black | White | Empty deriving (Eq, Show)
type Position = (Int, Int)
type Board = Map.Map Position Chess

--Below are functions that help output the board info
printChess :: Chess -> String
printChess chess =
    case chess of
        Black -> "X"
        White -> "O"
        Empty -> "_"

printBoard :: Board -> String
printBoard board = "\n  0 1 2 3 4 5 6 7 \n" ++ (intercalate "\n" (map (printRow board) [0..7])) ++
                   "\n  0 1 2 3 4 5 6 7 \n \n"
                 where
                    printRow b row =  show row  ++ " " ++ (intercalate " " (
                       map (\position -> printChess (fromMaybe Empty (Map.lookup position b)))
                        ([(row, x) | x <- [0..7]]))) ++ " " ++ show row

--Below are some general helper functions
addPosition :: Position -> Position -> Position
addPosition (x1, y1) (x2, y2) = (x1 + x2, y1 + y2)

opponent :: Chess -> Chess
opponent Black = White
opponent White = Black
opponent _ = Empty

--In order to be a legal move, the chess has to be able to flip some other chess
--and it should be placed in a legal empty place.
isLegal :: Chess -> Board -> Position -> Bool
isLegal color board position
    | not (position `elem` allPositions)  = False
    | fromMaybe Empty (Map.lookup position board) == Empty  =  (flipChessInMove color board position) /= []
    | otherwise                           = False
        where
            allPositions = [(x, y) | x <- [0..7], y <- [0..7]]

legalMoves :: Chess -> Board -> [Position]
legalMoves color board = filter (isLegal color board) allPositions
        where
            allPositions = [(x, y) | x <- [0..7], y <- [0..7]]

--Below are functions that return the possible flips
--For example, if I am putting a white chess, there must be a direction that starts with the white chess,
--and then black chess (can be one or more) and another white chess.
flipChessInMove :: Chess -> Board -> Position -> [Position]
flipChessInMove color board position
  | null $ changed = []
  | otherwise      = position : changed
  where
    directions = [(0, 1), (1, 0), (1, 1), (0, -1), (-1, 0), (-1, -1), (-1, 1), (1, -1)]
    changed = concat (map (flipChessInOneDir True color board position) directions)

flipChessInOneDir :: Bool -> Chess -> Board -> Position -> Position -> [Position]
flipChessInOneDir origin color board position direction
         | nextColor == opponent color =   if rest /= [] then
                                            --Check to see if this is the exact place we plan to put the chess or not
                                                if not origin  then
                                                    position : rest
                                                 else rest
                                            else []
         | nextColor == color          =    if not origin then
                                                [position]
                                            else []
         | otherwise                   =    []
      where
        nextPosition = addPosition position direction
        nextColor = fromMaybe Empty (Map.lookup nextPosition board)
        rest = flipChessInOneDir False color board nextPosition direction

move :: Chess -> Board -> Position -> Board
move color board position = Map.union (Map.fromList
                            (zip (flipChessInMove color board position) (repeat color))) board

--Below are functions that implement the Minimax WITH Alpha Beta Pruning strategy
advantage :: Chess -> Board -> Int
advantage color board =  sum (map (\(_, x) -> differ x) (Map.toList board))
            where
                    differ x
                      | x == color = 1
                      | x == Empty = 0
                      | otherwise = -1

alphaBetaPruning :: Int -> Chess -> Board -> Int -> Int -> Bool -> Int
alphaBetaPruning depth color board alpha beta maximizing
  | legalOpponentMoves == []  = advantage color board
  | depth == 0                = advantage color board
  | maximizing == True        = last alphaBetaForMinimizing
  | otherwise                 = last alphaBetaForMaximizing
  where
    opponentColor = opponent color
    legalOpponentMoves = legalMoves opponentColor board
    alphaBetaForMinimizing = (takeWhile (<beta))(scanl f alpha legalOpponentMoves)
       where f = \value position -> max value (alphaBetaPruning (depth-1) opponentColor (move opponentColor board position) value beta False)

    alphaBetaForMaximizing = (takeWhile (> alpha))(scanl g beta legalOpponentMoves)
       where g = \value position -> min value (alphaBetaPruning (depth-1) opponentColor (move opponentColor board position) alpha value True)


-- AI will make the best move according to the minMaxAdvantage strategy
aiMove ::  Int -> Chess -> Board -> Position
aiMove depth color board =
  (\(x, _) -> x)
    (maximumBy
      (\(_, x) (_, y) -> compare x y)
      (map
        (\position -> (position, alphaBetaPruning depth color (move color board position) (-100000) 100000 True))
        (legalMoves color board)))

-- The game logistic
game :: Chess -> Board -> IO()
game color board = do
  let gameOver = legalMoves color board == []
  if gameOver
    then
      do
        putStr (printBoard board)
        putStr "Game Over\n"

        if advantage color board == 0
          then
            putStr "Tie"
          else
            if advantage color board > 0
              then putStr ("Color " ++ (printChess color) ++ " won.")
              else putStr ("Color " ++ (printChess (opponent color)) ++ " won.")
    else
      do
        putStr (printBoard board)
        if color == White then
            -- Player move.
            do
              putStr ("Make your move, using format (x,y): \n")
              position <- readLn
              if not (isLegal color board position) then
                   do
                      putStr "Illegal move. Please try again \n"
                      game color board
              else
                   makeMove position
        else
             -- AI move. Assume recursive depth as 6
            do
               putStr ("AI is deciding its move. \n")
               makeMove (aiMove 6 color board)

         where
             makeMove position =
                do
                    game (opponent color) (move color board position)

-- The game starts with user's move (White chess)
main :: IO()
main = do game White initialBoard
        where
            -- According to the rul1e, we should start with four chess in the middle
            initialBoard = (Map.fromList [((3, 3), White), ((4, 4), White), ((3, 4), Black),((4, 3), Black)])
