import Data.List as L
import Data.Maybe
import qualified Data.Map as M
import Control.Parallel.Strategies(using, parList, rseq)

data Color = White | Black | Empty deriving (Eq, Show)
type Pos = (Int, Int)
type Board = M.Map Pos Color

flipC :: Color -> Color
flipC White = Black
flipC Black = White
flipC _ = Empty

allMoves :: Color -> Board -> [Pos]
allMoves color board = filter (isLegal color board) [(x, y) | x <- [0..7], y <- [0..7]]
  where isLegal color board pos = cellsChanged color board pos /= [] 
                                  && isNothing (M.lookup pos board)

cellsChanged :: Color -> Board -> Pos -> [Pos]
cellsChanged color board pos
  | null flipped = []
  | otherwise    = pos : flipped
  where flipped  = concatMap (rowChange True color board pos) 
                [(0, 1), (1, 1), (1, 0), (1, -1), (0, -1), (-1, -1), (-1, 0), (-1, 1)]
        rowChange isFirst color board pos dir
          | nextColor == Just (flipC color) = case restOfRow of
                                              []     -> []
                                              (x:xs) -> if isFirst then restOfRow 
                                                        else pos : restOfRow
          | nextColor == Just color = [pos | not isFirst]
          | otherwise = []
          where nextPos = (\(x, y) (dx, dy)  -> (x + dx, y + dy)) pos dir
                nextColor = M.lookup nextPos board
                restOfRow = rowChange False color board nextPos dir

advCount :: Color -> Board -> Int
advCount color board = sum $ map (\(_, x) -> advPerCell x) $ M.toList board
  where
    advPerCell x
      | x == color = 1
      | x == Empty = 0
      | otherwise = -1

heuristic :: Color -> Board -> Int
heuristic color board = advCount color board + 20 * optCountAdv color board
  where 
    optCountAdv :: Color -> Board -> Int
    optCountAdv color board = optCounts color board - optCounts (flipC color) board
    optCounts cl bd = length $ allMoves cl bd

step :: Color -> Board -> Pos -> Board
step color board pos = M.union 
  (M.fromList (zip (cellsChanged color board pos) (repeat color))) board

optMove :: Color -> Board -> Pos
optMove color board =
  fst $ maximumBy (\(_, x) (_, y) -> compare x y)
      (map (\pos -> (pos, miniMax 3 color (step color board pos)))
      (allMoves color board))

miniMax :: Int -> Color -> Board -> Int
miniMax depth color board
  | gameOver = if advCount color board > 0
    then 10000000
    else -10000000
  | depth <= 0 = heuristic color board
  | otherwise = if nc /= color
    then -maxAdvOp
    else  maxAdvOp
  where
    opMoves = allMoves (flipC color) board
    moves = allMoves color board
    gameOver = null moves && null opMoves
    nc = if opMoves /= [] then flipC color else color
    ncMoves = if nc /= color then opMoves else moves
    maxAdvOp = maximum (map
               (miniMax (depth - 1) nc . step nc board)
               ncMoves `using` parList rseq)

colorToChar :: Color -> String
colorToChar Empty = " "
colorToChar White = "O"
colorToChar Black = "X"

renderBoard :: Board -> String
renderBoard board =
  "\n    0   1   2   3   4   5   6   7 \n  +---+---+---+---+---+---+---+---+\n" ++
  intercalate "\n  +---+---+---+---+---+---+---+---+\n" (map (renderRow board) [0 .. 7])
  ++ "\n  +---+---+---+---+---+---+---+---+\n"
  where renderRow board row = show row ++ " | " ++
          intercalate " | " [helper (x, row) | x <- [0 .. 7]] ++ " | "
        helper position = colorToChar (fromMaybe Empty (M.lookup position board))

gameOver :: Color -> Int -> IO ()
gameOver color advCount
  | advCount == 0 = putStr "Game tie\n"
  | advCount > 0 = putStr (colorToChar color ++ " won by " ++ 
                          show advCount ++ "\n")
  | otherwise = putStr (colorToChar (flipC color) ++ " won by " ++ 
                          show (-advCount) ++ "\n")

go :: Color -> Board -> IO ()
go color board = 
  if null (allMoves color board) && null (allMoves (flipC color) board)
  then gameOver color $ advCount color board
  else do
    let oc = flipC color
        move = optMove color board
        nb = step color board move
        nc = if allMoves oc nb /= [] then oc else color
    putStr (renderBoard board)
    go nc nb

main :: IO ()
main = go White newBoard
  where newBoard = M.fromList 
                  [((3, 3), White), ((4, 4), White), ((3, 4), Black), ((4, 3), Black)]