module Board
  ( Color(..)
  , Point(..)
  , Board(..)
  , initBoard
  , oppositeColor
  , filterBoard
  , isEmptyBoard
  , addPoint
  , isValidPoint
  , isVacant
  , checkWin
  , getCurPoint
  ) where

import Data.List

data Color
  = Black
  | White
  | Empty
  deriving (Eq)

instance Show Color where
  show Black = "X"
  show White = "O"
  show Empty = "_"

data Point =
  Point
    { color    :: Color
    , position :: (Int, Int)
    }

instance Show Point where
  show (Point color _) = show color

instance Eq Point where
  (Point color1 (x1, y1)) == (Point color2 (x2, y2)) = x1 == x2 && y1 == y2 && color1 == color2

instance Ord Point where
  compare (Point _ (x1,y1)) (Point _ (x2,y2)) = compare (x1*10+y1) (x2*10+y2)

newtype Board = Board [[Point]]

instance Show Board where
  show (Board points) = intercalate "\n" $ map show points

instance Eq Board where
  (Board points1) == (Board points2) = points1 == points2

initBoard :: Board
initBoard = Board points
  where
    points = [initRow x 10 | x <- [1 .. 10]]
    initRow _ 0 = []
    initRow row col = initRow row (col - 1) ++ [Point Empty (row, col)]

getPoint :: Board -> (Int, Int) -> Point
getPoint (Board points) (x, y) = (points !! (x - 1)) !! (y - 1)

isValidPoint :: Point -> Bool
isValidPoint (Point _ (x, y))
  | x > 0 && x <= 10 && y > 0 && y <= 10 = True
  | otherwise = False

isVacant :: Point -> Board -> Bool
isVacant (Point color (x, y)) (Board points) = curColor == Empty
  where
    (Point curColor (_, _)) = getPoint (Board points) (x, y)

addPoint :: Board -> Color -> Int -> Int -> Board
addPoint (Board points) color x y
  | isValidPoint (Point color (x, y)) && isVacant (Point color (x, y)) (Board points) =
    add (Point color (x, y)) (Board points)
  | otherwise = Board points

add :: Point -> Board -> Board
add (Point color (newx, newy)) (Board points) = Board newPoints
  where
    newPoints = upperRows ++ (leftCells ++ (Point color (newx, newy) : rightCells)) : lowerRows
    (upperRows, thisRow:lowerRows) = splitAt (newx - 1) points
    (leftCells, _:rightCells) = splitAt (newy - 1) thisRow

checkWin :: Point -> Board -> Color
checkWin (Point color (x, y)) (Board points)
  | winRow (Point color (x, y)) (Board points) /= 0 ||
      (winCol (Point color (x, y)) (Board points) /= 0) ||
      (winDiag (Point color (x, y)) (Board points) /= 0) || (winAntiDiag (Point color (x, y)) (Board points) /= 0) =
    color
  | otherwise = Empty

checkRow :: [Point] -> Color -> Int -> Int
checkRow [] preColor cnt =
  if cnt == 5
    then if preColor == Black
           then 1
           else 2
    else 0
checkRow (head:xs) preColor cnt
  | preColor == Empty = checkRow xs color 1
  | preColor == color && cnt < 4 = checkRow xs color (cnt + 1)
  | preColor == color && cnt == 4 =
    if color == Black
      then 1
      else 2
  | otherwise = 0
  where
    (Point color _) = head

getDiag :: Board -> Board
getDiag (Board points) = Board $ diagonals points

getAntiDiag :: Board -> Board
getAntiDiag (Board points) = Board $ diagonals ((transpose . reverse) points)

diagonals :: [[a]] -> [[a]]
diagonals = tail . go []
  where
    go b es_ =
      [h | h:_ <- b] :
      case es_ of
        []   -> transpose ts
        e:es -> go (e : ts) es
      where
        ts = [t | _:t <- b]

winRow :: Point -> Board -> Int
winRow (Point color (x, y)) (Board points) = checkRow (newPoints !! (x - 1)) Empty 1
  where
    Board newPoints = addPoint (Board points) color x y

winCol :: Point -> Board -> Int
winCol (Point color (x, y)) (Board points) = checkRow ((transpose . reverse) newPoints !! (y - 1)) Empty 1
  where
    Board newPoints = addPoint (Board points) color x y

winDiag :: Point -> Board -> Int
winDiag (Point color (x, y)) (Board points) = checkRow (diagonals newPoints !! (x + y - 2)) Empty 1
  where
    Board newPoints = addPoint (Board points) color x y

winAntiDiag :: Point -> Board -> Int
winAntiDiag (Point color (x, y)) (Board points) =
  checkRow (diagonals ((transpose . reverse) newPoints) !! (9 - x + y)) Empty 1
  where
    Board newPoints = addPoint (Board points) color x y

isEmptyBoard :: Board -> Bool
isEmptyBoard (Board points) = Board points == initBoard

oppositeColor :: Color -> Color
oppositeColor color
  | color == White = Black
  | color == Black = White
  | otherwise = error "Invalid opposite color"
  
filterBoard :: Board -> Color -> [Point]
filterBoard (Board points) color =
  [p | rows <- points, p <- rows, isSameColor p]
  where
    isSameColor (Point c (_,_)) = c == color

flatten :: [[a]] -> [a]
flatten xs = (\z n -> foldr (flip (foldr z)) n xs) (:) []

getCurPoint :: Board -> Board -> [Point]
getCurPoint (Board points1) (Board points2) = flatten points2 \\ flatten points1
