module AI
  ( moveAI
  ) where

import           Board
import           Control.Parallel.Strategies
import           Data.List
import           Data.Maybe
import qualified Data.Set                    as Set
import           Data.Tree

minInt :: Int
minInt = -(2 ^ 29)

maxInt :: Int
maxInt = 2 ^ 29 - 1

moveAI :: Board -> Color -> Board
moveAI board color
  | isEmptyBoard board = addPoint board color 1 1
  | otherwise = bestMove
  where
    neighbors = possibleMoves board
    (Node node children) = buildTree color board neighbors
    minmax = parMap rdeepseq (minmaxBeta color 3 minInt maxInt) children
    index = fromJust $ elemIndex (maximum minmax) minmax
    (Node bestMove _) = children !! index

buildTree :: Color -> Board -> [Point] -> Tree Board
buildTree color board neighbors = Node board $ children neighbors
  where
    newNeighbors point =
      Set.toList $
      Set.union (Set.fromList (Data.List.delete point neighbors)) (Set.fromList (stepFromPoint board point))
    oppoColor = oppositeColor color
    children [] = []
    children (Point c (x, y):ns) =
      buildTree oppoColor (addPoint board color x y) (newNeighbors (Point c (x, y))) : children ns

minmaxAlpha :: Color -> Int -> Int -> Int -> Tree Board -> Int
minmaxAlpha _ _ alpha _ (Node _ []) = alpha
minmaxAlpha color level alpha beta (Node b (x:xs))
  | level == 0 = curScore
  | canFinish curScore = curScore
  | newAlpha >= beta = beta
  | otherwise = minmaxAlpha color level newAlpha beta (Node b xs)
  where
    curScore = scoreBoard b color
    canFinish score = score > 100000 || score < (-100000)
    newAlpha = maximum [alpha, minmaxBeta color (level - 1) alpha beta x]

minmaxBeta :: Color -> Int -> Int -> Int -> Tree Board -> Int
minmaxBeta _ _ _ beta (Node _ []) = beta
minmaxBeta color level alpha beta (Node b (x:xs))
  | level == 0 = curScore
  | canFinish curScore = curScore
  | alpha >= newBeta = alpha
  | otherwise = minmaxBeta color level alpha newBeta (Node b xs)
  where
    curScore = scoreBoard b color
    canFinish score = score > 100000 || score < (-100000)
    newBeta = minimum [beta, minmaxAlpha color (level - 1) alpha beta x]

scoreBoard :: Board -> Color -> Int
scoreBoard board color = score (pointsOfColor color) - score (pointsOfColor $ oppositeColor color)
  where
    score points = sum $ map sumScores $ scoreDirections points
    pointsOfColor = filterBoard board

sumScores :: [Int] -> Int
sumScores [] = 0
sumScores (x:xs)
  | x == 5 = 100000 + sumScores xs
  | x == 4 = 10000 + sumScores xs
  | x == 3 = 1000 + sumScores xs
  | x == 2 = 100 + sumScores xs
  | otherwise = sumScores xs

scoreDirections :: [Point] -> [[Int]]
scoreDirections [] = [[0]]
scoreDirections ps@(point:rest) =
  parMap
    rdeepseq
    (scoreDirection point ps 0)
    [(xDir, yDir) | xDir <- [0 .. 1], yDir <- [-1 .. 1], not (xDir == 0 && yDir == (-1)), not (xDir == 0 && yDir == 0)]

scoreDirection :: Point -> [Point] -> Int -> (Int, Int) -> [Int]
scoreDirection _ [] cont (_, _) = [cont]
scoreDirection (Point c (x, y)) ps@(Point c1 (x1, y1):rest) cont (xDir, yDir)
  | Point c (x, y) `elem` ps =
    scoreDirection (Point c (x + xDir, y + yDir)) (Data.List.delete (Point c (x, y)) ps) (cont + 1) (xDir, yDir)
  | otherwise = cont : scoreDirection (Point c1 (x1, y1)) rest 1 (xDir, yDir)

possibleMoves :: Board -> [Point]
possibleMoves board = Set.toList $ stepBoard board $ filterBoard board White ++ filterBoard board Black

stepBoard :: Board -> [Point] -> Set.Set Point
stepBoard _ [] = Set.empty
stepBoard board (point:rest) = Set.union (Set.fromList (stepFromPoint board point)) $ stepBoard board rest

stepFromPoint :: Board -> Point -> [Point]
stepFromPoint board (Point _ (x, y)) =
  [ Point Empty (x + xDir, y + yDir)
  | xDir <- [-1 .. 1]
  , yDir <- [-1 .. 1]
  , not (xDir == 0 && yDir == 0)
  , isValidPoint (Point Empty (x + xDir, y + yDir))
  , isVacant (Point Empty (x + xDir, y + yDir)) board
  ]
