import Control.Monad(forM_, forever)
import Control.Concurrent.STM
import Control.Concurrent(forkIO, forkFinally, threadDelay)
import Control.Parallel
import Control.Parallel.Strategies(parMap, rpar)
import System.Directory (doesDirectoryExist, getDirectoryContents)
import System.FilePath ((</>))
import qualified Data.ByteString.Char8 as B
import qualified Data.ByteString.Search as BS
import System.Environment(getArgs, getProgName)

numOfWorkers = 4


grabFiles :: FilePath -> Bool -> TChan (Maybe FilePath) -> IO ()
grabFiles fpath recursive chan = do
    walkDir fpath recursive
    -- add terminators
    forM_ [0..numOfWorkers-1] $ \_ -> atomically $ writeTChan chan Nothing
  where
    fileFilter fname = head fname /= '.'
    walkDir :: FilePath -> Bool -> IO ()
    walkDir path recursive = do
        isDir <- doesDirectoryExist path
        if isDir
            then do
                names <- getDirectoryContents path
                let properNames = filter fileFilter names
                forM_ properNames $ \fname -> walkDir (path </> fname) recursive
            else atomically $ writeTChan chan (Just path)


runGrap :: String -> FilePath -> IO ()
runGrap pat filepath = do
    jobChan <- newTChanIO
    outChan <- newTChanIO
    let bpat = B.pack pat

    forkIO $ grabFiles filepath True jobChan

    -- start workers
    forM_ [0..numOfWorkers-1] $ \i -> forkIO $ runWorker i jobChan outChan bpat

    -- gather result and print
    printResults outChan


runWorker :: Int -> TChan (Maybe FilePath) -> TChan Output -> B.ByteString -> IO ()
runWorker wid jobChan outChan pat = runLoop
    where
        runLoop = do
            filename <- atomically $ readTChan jobChan
            case filename of
                Just fname -> do
                    searchInFile pat fname outChan
                    runLoop
                Nothing -> atomically $ writeTChan outChan Terminated

searchInFile :: B.ByteString -> FilePath -> TChan Output -> IO()
searchInFile pat fname outChan = do
    content <- B.readFile fname

    let augLines = zip [1..] $ B.lines content
        matches = filter (\al@(_, line) -> not . null $ BS.indices pat line) augLines
    atomically $ writeTChan outChan (Matches fname matches)

printResults :: TChan Output -> IO ()
printResults outChan = loop 0
    where
        loop i =
            if i == numOfWorkers
                then return ()
            else do
                output <- atomically $ readTChan outChan
                case output of
                    Terminated -> loop (i + 1)
                    Matches fpath results -> do
                        forM_ results $ \(ln, txt) -> putStrLn $ fpath ++ ":" ++ show ln ++ ": " ++ B.unpack txt
                        loop i


data Output = Terminated | Matches FilePath [(Int, B.ByteString)]


main :: IO ()
main = do
    [pat, filename] <- getArgs
    runGrap pat filename
