/*
 * Userspace program that communicates with the vga_ball device driver
 * through ioctls
 *
 * Stephen A. Edwards
 * Columbia University
 */

#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>


#include "midi_top.h"
//int midi_top_fd;

unsigned short active_notes [10];
int available [10];

void note_init() {
  memset(active_notes, 0, 10*sizeof(short));
  memset(available, -1, 10*sizeof(int));
}

void start_wave(int midi_top_fd, unsigned short data)
{
  midi_top_arg_t vla;
  vla.note.note_info = data;
  
  if (ioctl(midi_top_fd, MIDI_TOP_START_WAVE, &vla)) {
      perror("ioctl(MIDI_TOP_START_WAVE) failed");
      return;
  }
}

void send_wave(int midi_top_fd, short data)
{
  midi_top_arg_t vla;
  vla.note.note_info = data;
  
  if (ioctl(midi_top_fd, MIDI_TOP_SEND_WAVE, &vla)) {
      perror("ioctl(MIDI_TOP_SEND_WAVE) failed");
      return;
  }
}


//make sure midi_number >=12 (startin at C0)
// WRITTEN FOR TONEGEN -- we will need to make changes later
// assume volume 0-127
unsigned short to_note_info(unsigned short midi_number, unsigned short volume)
{
  unsigned short note_info;

  assert(midi_number >= 12 && midi_number <= 119);
  assert(volume < 128);

  unsigned short note, octave;
  note   = midi_number % 12;
  octave = (midi_number / 12) - 1;

  // note - 4 bits, from 0-11 corresponding to C -> B
  // octave - 4 bits, from 0-8 
  // volume - 0-127 for 
  
  unsigned int status = (volume != 0);
  
  printf("note= %d, octave= %d, volume= %d\n", note, octave, volume);
  note_info = volume;
  note_info |= (status << 7);
	
  note_info |= (note   << 8);
  note_info |= (octave << 12);

  return note_info;
}


void display_note(unsigned short note_info)
{
  unsigned short note, octave, volume;

  volume = note_info & 127;
  note   = (note_info >> 8) & 0x000F;
  octave = note_info >> 12;
  
  printf("note= %d, octave= %d, volume= %d\n", note, octave, volume);
}


void set_note(int midi_top_fd, unsigned short midi_number, unsigned short volume)
{
  // search for midi note number in active_notes
  int index = 0;
  int found = 0;
  while(index < 10 && !found) {
    if (midi_number == active_notes[index++]) found = 1;
  }
  
  int status = (volume != 0);
  if(found) {
    // just update the existing entry
    available[index] = !status;
  } else {
    // search for a free space
    if (status) {
      index = 0;
      while(index < 10 && !found) {
        if (available[index++]) found = 1;
      }
    }
  }
  
  // prepare write data
  midi_top_arg_t vla;
  vla.note.note_info = to_note_info(midi_number, volume);
  vla.note.index = index;

  // sent to the hardware
  if (found) {
    if (ioctl(midi_top_fd, MIDI_TOP_WRITE_NOTE, &vla)) {
        perror("ioctl(MIDI_TOP_WRITE_NOTE) failed");
        return;
    }
    printf("SENT: ");
    display_note(vla.note.note_info);
  }
}


/* Read and print the background color */
void print_note(int midi_top_fd) {
  midi_top_arg_t vla;
  
  if (ioctl(midi_top_fd, MIDI_TOP_READ_NOTE, &vla)) {
      perror("ioctl(MIDI_TOP_READ_NOTE) failed");
      return;
  }
  display_note(vla.note.note_info);
  //printf("%02x %02x %02x\n",
	// vla.background.red, vla.background.green, vla.background.blue);
}

/*
int main()
{


  //printf("MIDI top Userspace program started\n");


  int midi_top_fd;
  static const char filename[] = "/dev/midi_top";

  if ( (midi_top_fd = open(filename, O_RDWR)) == -1) {
    fprintf(stderr, "could not open %s\n", filename);
    return -1;
  }

  set_note(midi_top_fd,60, 127);

  //printf("MIDI top Userspace program terminating\n");
  return 0;
}
*/
