/* * Device driver for the VGA video generator
 *
 * A Platform device implemented using the misc subsystem
 *
 * Stephen A. Edwards
 * Columbia University
 *
 * References:
 * Linux source: Documentation/driver-model/platform.txt
 *               drivers/misc/arm-charlcd.c
 * http://www.linuxforu.com/tag/linux-device-drivers/
 * http://free-electrons.com/docs/
 *
 * "make" to build
 * insmod vga_ball.ko
 *
 * Check code style with
 * checkpatch.pl --file --no-tree midi_top.c
 */

#include "midi_top.h"
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/version.h>

#define DRIVER_NAME "midi_top"

/* Device registers */
#define REG_NOTE(x) (x)
#define WAVE_START(x) ((x) + 20)
#define WAVE_DATA(x) ((x) + 22)
#define MODULATION(x) ((x) + 24)

//#define BG_GREEN(x) ((x)+1)
//#define BG_BLUE(x) ((x)+2)

/*
 * Information about our device
 */
struct midi_top_dev {
  struct resource res;    /* Resource: our registers */
  void __iomem *virtbase; /* Where registers can be accessed in memory */
  midi_top_note_t note;
} dev;

static void
write_start_wave(midi_top_note_t *note) { // unsigned short table_number) {
  iowrite16(note->note_info, WAVE_START(dev.virtbase));
}

static void
write_send_wave(midi_top_note_t *note) { // unsigned short wave_data) {
  iowrite16(note->note_info, WAVE_DATA(dev.virtbase));
}

/*
 * Write segments of a single digit
 * Assumes digit is in range and the device information has been set up
 */
static void write_note(midi_top_note_t *note) {
  iowrite16(note->note_info, dev.virtbase + (2 * note->index));
  dev.note = *note;
}

/*
 * Handle ioctl() calls from userspace:
 * Read or write the segments on single digits.
 * Note extensive error checking of arguments
 */
static long midi_top_ioctl(struct file *f, unsigned int cmd,
                           unsigned long arg) {
  midi_top_arg_t vla;

  switch (cmd) {

  case MIDI_TOP_START_WAVE:
    if (copy_from_user(&vla, (midi_top_note_t *)arg, sizeof(midi_top_note_t)))
      return -EACCES;
    write_start_wave(&vla.note);
    // write_note(&vla.note);
    // write_background(&vla.background);
    break;

  case MIDI_TOP_SEND_WAVE:
    if (copy_from_user(&vla, (midi_top_note_t *)arg, sizeof(midi_top_note_t)))
      return -EACCES;
    write_send_wave(&vla.note);
    // write_note(&vla.note);
    // write_background(&vla.background);
    break;

  case MIDI_TOP_WRITE_NOTE:
    if (copy_from_user(&vla, (midi_top_note_t *)arg, sizeof(midi_top_note_t)))
      return -EACCES;
    write_note(&vla.note);
    // write_background(&vla.background);
    break;

  case MIDI_TOP_READ_NOTE:
    vla.note = dev.note;
    if (copy_to_user((midi_top_note_t *)arg, &vla, sizeof(midi_top_note_t)))
      return -EACCES;
    break;

  default:
    return -EINVAL;
  }

  return 0;
}

/* The operations our device knows how to do */
static const struct file_operations midi_top_fops = {
    .owner = THIS_MODULE,
    .unlocked_ioctl = midi_top_ioctl,
};

/* Information about our device for the "misc" framework -- like a char dev */
static struct miscdevice midi_top_misc_device = {
    .minor = MISC_DYNAMIC_MINOR,
    .name = DRIVER_NAME,
    .fops = &midi_top_fops,
};

/*
 * Initialization code: get resources (registers) and display
 * a welcome message
 */
static int __init midi_top_probe(struct platform_device *pdev) {
  // midi_top_color_t beige = { 0xf9, 0xe4, 0xb7 };
  int ret;

  /* Register ourselves as a misc device: creates /dev/midi_top */
  ret = misc_register(&midi_top_misc_device);

  /* Get the address of our registers from the device tree */
  ret = of_address_to_resource(pdev->dev.of_node, 0, &dev.res);
  if (ret) {
    ret = -ENOENT;
    goto out_deregister;
  }

  /* Make sure we can use these registers */
  if (request_mem_region(dev.res.start, resource_size(&dev.res), DRIVER_NAME) ==
      NULL) {
    ret = -EBUSY;
    goto out_deregister;
  }

  /* Arrange access to our registers */
  dev.virtbase = of_iomap(pdev->dev.of_node, 0);
  if (dev.virtbase == NULL) {
    ret = -ENOMEM;
    goto out_release_mem_region;
  }

  /* Set an initial color */
  // write_background(&beige);

  return 0;

out_release_mem_region:
  release_mem_region(dev.res.start, resource_size(&dev.res));
out_deregister:
  misc_deregister(&midi_top_misc_device);
  return ret;
}

/* Clean-up code: release resources */
static int midi_top_remove(struct platform_device *pdev) {
  iounmap(dev.virtbase);
  release_mem_region(dev.res.start, resource_size(&dev.res));
  misc_deregister(&midi_top_misc_device);
  return 0;
}

/* Which "compatible" string(s) to search for in the Device Tree */
#ifdef CONFIG_OF
static const struct of_device_id midi_top_of_match[] = {
    {.compatible = "csee4840,midi_top-1.0"},
    {},
};
MODULE_DEVICE_TABLE(of, midi_top_of_match);
#endif

/* Information for registering ourselves as a "platform" driver */
static struct platform_driver midi_top_driver = {
    .driver =
        {
            .name = DRIVER_NAME,
            .owner = THIS_MODULE,
            .of_match_table = of_match_ptr(midi_top_of_match),
        },
    .remove = __exit_p(midi_top_remove),
};

/* Called when the module is loaded: set things up */
static int __init midi_top_init(void) {
  pr_info(DRIVER_NAME ": init\n");
  return platform_driver_probe(&midi_top_driver, midi_top_probe);
}

/* Calball when the module is unloaded: release resources */
static void __exit midi_top_exit(void) {
  platform_driver_unregister(&midi_top_driver);
  pr_info(DRIVER_NAME ": exit\n");
}

module_init(midi_top_init);
module_exit(midi_top_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Stephen A. Edwards, Columbia University");
MODULE_DESCRIPTION("MIDI top driver");
