module wavetables(
	input  logic 		clk,
	input  logic 		rst,

	input  logic 		write,     // set to true if linux sent us something
	input  logic 		is_data,   // 0 is info, 1 is wave data
	input  logic [15:0] wave_data, // wave data

	input  logic [15:0] sample_address,

	input  logic 		sample_ready,
	output logic 		sample_valid,
	
	input logic [6:0] mix_ratio,
	output logic signed [15:0] data_out
);

logic [15:0] sample_data0;
logic [15:0] sample_data1;

logic  [15:0] a_wave, a_0, a_1, a_2;
//logic  [15:0] din_wave;
logic         we_wave, we_0, we_1, we_2;
logic  [15:0] dout_0, dout_1, dout_2;

//logic switch_buffers;
logic wave_num;
logic start;
logic done;

//assign switch_buffers = done;

//"S021" is State(write = 0, read0 = 2, read1 = 1) - writing to buffer 0 from software, sampledata0 = table2, sampledata1 = table1
typedef enum logic[5:0] {S012, S021, S102, S120, S201, S210} wave_state_t;
//wave_state_t wave_state, wave_next_state;
wave_state_t wave_state;
wave_state_t wave_next_state;

mixer mx (.reset(rst), .wave1(sample_data0), .wave2(sample_data1), .mix_ratio(mix_ratio), .wave_out(data_out));

memory table0(.clk(clk),
			  .a(a_0),
			  .din(wave_data),
			  .we(we_0),
			  .dout(dout_0));

memory table1(.clk(clk),
			  .a(a_1),
			  .din(wave_data),
			  .we(we_1),
			  .dout(dout_1));

memory table2(.clk(clk),
			  .a(a_2),
			  .din(wave_data),
			  .we(we_2),
			  .dout(dout_2));

always_ff @(posedge clk or posedge rst) begin
	we_wave				<= 0; //default
	done                <= 0;
	sample_valid <= 0;
	if (rst) begin
		start           <= 0;
		wave_state		<= S012;
		wave_num		<= 0;
		a_wave			<= 0;
	end else begin
		if (sample_ready == 1)
			sample_valid <= 1;
		wave_state <= wave_next_state;
		if (write) begin
			if ( is_data == 0) begin
				wave_num		<= wave_data[0]; //info bit - which sample to replace?
				start           <= 1;
				a_wave			<= 0;
			end else begin
				if ( a_wave > 47998 ) begin
					we_wave <= 0;
				end else begin
					start <= 0;
					if (a_wave == 47998 ) begin
						a_wave  <= a_wave + 1;
						done    <= 1;
						we_wave <= 1;
					end else begin
						we_wave <= 1;
						a_wave  <= a_wave + 1;
					end
				end
			end
		end
	end
end


// State Machine to determine
always_comb begin
	wave_next_state = wave_state; //default;
	case (wave_state)
		S012: begin
			if (done)
				wave_next_state = (wave_num ? S210 : S102);

			a_0    = a_wave;
			we_0   = we_wave;

			a_1    = sample_address;
			a_2    = sample_address;
			we_1   = 0;
			we_2   = 0;

			sample_data0 = dout_1;
			sample_data1 = dout_2;
		end
		S021: begin
			if (done)
				wave_next_state = (wave_num ? S120 : S201);

			a_0    = a_wave;
			we_0   = we_wave;

			a_1    = sample_address;
			a_2    = sample_address;
			we_1   = 0;
			we_2   = 0;

			sample_data0 = dout_2;
			sample_data1 = dout_1;
		end
		S102: begin
			if (done)
				wave_next_state = (wave_num ? S201 : S012);
				
			a_1    = a_wave;
			we_1   = we_wave;

			a_0    = sample_address;
			a_2    = sample_address;
			we_0   = 0;
			we_2   = 0;

			sample_data0 = dout_0;
			sample_data1 = dout_2;
		end
		S120: begin
			if (done)
				wave_next_state = (wave_num ? S021 : S210);
				
			a_1    = a_wave;
			we_1   = we_wave;

			a_0    = sample_address;
			a_2    = sample_address;
			we_0   = 0;
			we_2   = 0;

			sample_data0 = dout_2;
			sample_data1 = dout_0;
		end
		S201: begin
			if (done)
				wave_next_state = (wave_num ? S102 : S021);
				
			a_2    = a_wave;
			we_2   = we_wave;

			a_0    = sample_address;
			a_1    = sample_address;
			we_0   = 0;
			we_1   = 0;

			sample_data0 = dout_0;
			sample_data1 = dout_1;
		end
		S210: begin
			if (done)
				wave_next_state = (wave_num ? S012 : S120);
				
			a_2    = a_wave;
			we_2   = we_wave;

			a_0    = sample_address;
			a_1    = sample_address;
			we_0   = 0;
			we_1   = 0;

			sample_data0 = dout_1;
			sample_data1 = dout_0;
		end
		//default: begin
		//	wave_next_state = 5'b11111;
		//end				
		endcase

end
endmodule



// 48000 X 16 synchronous RAM with old data read-during-write behavior
module memory( input logic         clk,
			   input logic  [15:0] a,
			   input logic  [15:0] din,
			   input logic         we,
			   output logic [15:0] dout);

logic [15:0]  mem [47999:0];
always_ff @(posedge clk) begin
	if (we) mem[a] <= din;
	dout <= mem[a];
end
	
endmodule 


module mixer (
		input reset,
		input logic signed [15:0] wave1,
		input logic signed [15:0] wave2,
		input logic [6:0] mix_ratio,
		output logic signed [15:0] wave_out
		);

always_comb begin
	logic signed [31:0] out1, out2;
	logic signed [15:0] rat;
	assign rat = {9'd0,mix_ratio};
	if (!reset) begin
		out1 = wave1 * rat;
		out2 = wave2 * (127-rat);
		wave_out = ((out1 >>> 7) + (out2 >>> 7));
		
	end	
	else 
		wave_out = 0;
end

endmodule 

