module note (
            // control
            input clk,
            input rst,
            input logic [15:0] settings ,
            // [15:12] = octave
            // [11:8] = note
            // [7] = pressed
            // [6:0] = volume

            // to/from summer
            output logic signed [15:0] sample_out,
            output logic [15:0] amplitude,
            output logic valid_out,
            input ready_in,

            // to/from arbiter
            input logic signed [15:0] sample_in,
            input valid_in,
            output logic [15:0] sample_address,
            output logic ready_out
        );

        logic [14:0] duration; // how long since note began
        logic [14:0] released_duration; // how long note has been released
        logic smpl_clk; // high each time a new sample is requested

        // for sampling logic
        enum integer {IDLE=0, READY=1, VALID=2} state;

        // octave 8 mem_offset values
	logic [15:0] notes [12] = '{//       +-------------------------
		4186, // C                   |          _______________
		4435, // C-sharp and D-flat  |---------|_______________
		4698, // D                   |          _______________
		4978, // D-sharp and E-flat  |---------|_______________
		5274, // E                   |_________________________
		5588, // F                   |          _______________
		5920, // F-sharp and G-flat  |---------|_______________
		6272, // G                   |          _______________
		6645, // G-sharp and A-flat  |---------|_______________
		7040, // A                   |          _______________
		7459, // A-sharp and B-flat  |---------|_______________
		7902  // B                   |
	};            //                     +-------------------------

        // settings
        logic [3:0] octave; // can only be 0-8, lower octaves are higher values
        logic [3:0] note; // can only be 0-11
        logic pressed; // is the note pressed down
        logic last_pressed; // value of pressed in last clock cycle
        logic [6:0] volume;
        logic [15:0] mem_offset; // function of octave and note
        assign octave = settings[15:12];
        assign note = settings[11:8];
        assign pressed = settings[7];
        assign volume = settings[6:0];
        assign mem_offset = notes[note] >> octave;

        logic [15:0] next_addr; // modified by mem_offset
        assign next_addr = sample_address + mem_offset;

        always_ff @(posedge clk) begin
            last_pressed <= pressed;

            // reset logic
            if(rst) begin
                state <= IDLE;
                sample_address <= 16'd0;
                ready_out <= 0;
                valid_out <= 0;
                duration <= 15'd0;
                released_duration <= 15'd0;
                smpl_clk <= 0;
            end else begin
                smpl_clk <= state == IDLE && ready_in;

                // reset dnote_sampleuration if the note was just pressed
                if (pressed & !last_pressed) begin
                    duration <= 15'd0;
                    released_duration <= 15'd0;
                end
                
                // sampling ready-valid handshake logic
                case (state)
                    //////////////////////////
                    IDLE : if (ready_in) begin
                               state <= READY;
                               ready_out <= 1;
                               valid_out <= 0;
                           end else begin
                               state <= IDLE;
                               ready_out <= 0;
                               valid_out <= 0;
                           end
                
                    //////////////////////////
                    READY: if (ready_in) begin
                               if (valid_in) begin
                                   state <= VALID;
                                   ready_out <= 0;
                                   valid_out <= 1;
                               end else begin
                                   state <= READY;
                                   ready_out <= 1;
                                   valid_out <= 0;
                               end
                           end else begin
                               state <= IDLE;
                               ready_out <= 0;
                               valid_out <= 0;
                           end

                    //////////////////////////
                    VALID: if (ready_in) begin
                               state <= VALID;
                               ready_out <= 0;
                               valid_out <= 1;
                           end else begin
                               state <= IDLE;
                               ready_out <= 0;
                               valid_out <= 0;
                           end
                endcase

                
                if (smpl_clk) begin
                    // update address
                    if (next_addr > 16'd47999) 
                        sample_address <= next_addr - 16'd48000;
                    else 
                        sample_address <= next_addr;
                    
                    // update durations
                    if (!pressed)
                        if (released_duration < 15'd9000)
                            released_duration <= released_duration + 15'd1;
                    if (duration < 15'd19000) 
                        duration <= duration + 15'd1;
               end
            end
        end

        // generating a sample
        adsr_envelope env(.*);

endmodule


// applies envelope effect on note
module adsr_envelope (
            input clk,
            input smpl_clk,
            input rst,
            input logic [14:0] duration,
            input logic [14:0] released_duration,
            input logic signed [15:0] sample_in,
            input logic [6:0] volume,
            input pressed,
            output logic signed [15:0] sample_out,
            output logic [15:0] amplitude
        );

        logic [9:0] attack_counter;
        logic [9:0] retreat_counter;
        logic [9:0] ac_div = 93;
        logic [9:0] dc_div = 300;
        logic [9:0] rc_div = 83;

        logic [7:0] envelope;
        logic signed [29:0] product;

        always_ff @(posedge clk) begin
            if (rst) begin
                attack_counter <= 0;
                retreat_counter <= 0;
                amplitude <= 0;
                sample_out <= 0;
                envelope <= 0;
            end else begin
                // reset when a new note is assigned
                if (duration == 15'd0) begin
                    attack_counter <= 0;
                    retreat_counter <= 0;
                end

                if (smpl_clk) begin
                    // increment counters
                    attack_counter <= attack_counter + 10'd1;
                    if (!pressed)
                        retreat_counter <= retreat_counter + 10'd1;
                    
                    // handle attack counter clock div values
                    if (duration < 15'd12000 && attack_counter == ac_div) begin
                        attack_counter <= 10'd0;
			envelope <=  envelope + 8'd1;
                    end else if (duration < 15'd18000 && attack_counter == dc_div) begin
                        attack_counter <= 10'd0;
                        // TODO 
						envelope <= envelope - 8'd1;
                    end

                    // handle retreat counter clock div values
                    if (released_duration < 15'd9000 && retreat_counter == rc_div) begin
                        retreat_counter <= 10'd0;
                        envelope <= envelope - 8'd1;
                    end
                end

                amplitude <= envelope * volume;
                product <= $signed(amplitude) * sample_in;              
		sample_out <= product / (1<<14);
            end
        end
endmodule
