/*
* Avalon memory-mapped peripheral that generates VGA
*
* Stephen A. Edwards
* Columbia University
*/

module midi_top(input logic clk,
		input logic 	    reset,
		input logic [15:0]  writedata,
		input logic         write,
		input               chipselect,
		input logic [3:0]   address,

		input [3:0]    KEY,
		
		input left_chan_ready,
		input right_chan_ready,
		output logic [15:0] sample_data,
		output logic sample_valid);


	//note top ports
	//from wave table
	//logic nt_wave_address_ready;
	//logic nt_wave_sample_valid;
	//logic signed [15:0] nt_wave_sample;
	//logic [15:0] nt_wave_address;
	
	//from MIDI top
	logic nt_note_data_valid;
	logic [15:0] nt_note_data;
	logic [3:0] nt_note_addr;
	//to audio codec    
	logic audio_codec_ready;
	assign audio_codec_ready = left_chan_ready & right_chan_ready;
	



	//wavetable ports
	logic 		 wt_write;     // set to true if linux sent us something
	logic 		 wt_is_data;   // 0 is info, 1 is wave data
	logic [15:0] wt_wave_data; // wave data

	logic [15:0] wt_sample_address;

	logic 		 wt_sample_ready;
	logic 	 	 wt_sample_valid;
	
	logic [6:0]  wt_mix_ratio;
	logic signed [15:0] wt_data_out;


	// logic [6:0] volume;
	// logic status_bit;
	// logic [3:0] note;
	// logic [3:0] octave;


	
	wavetables wt(.clk(clk), .rst(reset), 
	.write(wt_write), .is_data(wt_is_data) , .wave_data(wt_wave_data), 

	.sample_address(wt_sample_address),
	.sample_ready(wt_sample_ready) ,.sample_valid(wt_sample_valid),
	
	.mix_ratio(wt_mix_ratio), //set by avalon, below

	.data_out(wt_data_out)    
	);



	note_top nt(.clk(clk), .reset(reset), 

		.wave_address_ready(wt_sample_ready), .wave_sample_valid(wt_sample_valid),     // sent to/from wavetable, above
		.wave_sample(wt_data_out), .wave_address(wt_sample_address),

		.note_data_valid(nt_note_data_valid), .note_data(nt_note_data), .note_addr(nt_note_addr), // sent by avalon, below

		.audio_codec_ready(audio_codec_ready), .codec_sample_valid(sample_valid),                  //communicate with the codec
		.codec_sample_data(sample_data)
	);




	always_ff @(posedge clk) begin
		nt_note_data_valid <= 0;
		wt_write <= 0;
	 if (reset) begin
		nt_note_data_valid <= 0;
		wt_mix_ratio <= 0;
		wt_write <= 0;

	 end else if (chipselect && write) begin
	   case (address)
	 4'd0 : begin
		nt_note_addr       <= 0;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd1 : begin
		nt_note_addr       <= 1;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd2 : begin
		nt_note_addr       <= 2;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd3 : begin
		nt_note_addr       <= 3;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd4 : begin
		nt_note_addr       <= 4;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd5 : begin
		nt_note_addr       <= 5;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd6 : begin
		nt_note_addr       <= 6;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd7 : begin
		nt_note_addr       <= 7;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd8 : begin
		nt_note_addr       <= 8;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd9 : begin
		nt_note_addr       <= 9;
		nt_note_data       <= writedata;
		nt_note_data_valid <= 1;
		end
	 4'd10 : begin
	 	//START WAVE
	 	wt_write   <= 1;
	 	wt_is_data <= 0;
	 	wt_wave_data  <= writedata;
		end
	 4'd11 : begin
	 	//SEND WAVE
	 	wt_write   <= 1;
	 	wt_is_data <= 1;
	 	wt_wave_data  <= writedata;
		end
	 4'd12 : begin
	 	//MODULATION
	 	wt_mix_ratio <= writedata;
		end
	 default: begin
		wt_write   <= 0;
	 	wt_is_data <= 0;
	 	wt_wave_data  <= writedata;
		end
	endcase
     end
end

endmodule


// 48000 X 16 synchronous RAM with old data read-during-write behavior
/*
module memory( input logic         clk,
	input logic  [15:0] a,
	input logic  [15:0] din,
	input logic         we,
	output logic [15:0] dout);

logic [15:0]  mem [47999:0];

always_ff @(posedge clk) begin
	if (we) mem[a] <= din;
	dout <= mem[a];
end

endmodule
*/

